/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef _TOP10_UI_FRUSTUM
#define _TOP10_UI_FRUSTUM

#include "math/Vertex.hh"
#include "math/Plane.hh"
#include "math/Box.hh"
#include "helpers/GenericOctree.hh"
#include "helpers/PolygonSet.hh"
#include <math.h>

namespace top10 {
  namespace ui_interactive {
    //! Represent the viewing volume of a camera
    class Frustum {
    public:
      Frustum();
      Frustum(top10::math::Vector c, top10::math::Vector dir, top10::math::Vector up,
	      double r, double f, double n, double fa): center(c), ratio(r), fov(M_PI*f/180.0), dist_near(n), dist_far(fa)
      {
	setDirection(dir, up);
      }

      inline double getFOV() const { return fov*180.0/M_PI; }
      inline double getNear() const { return dist_near; }
      inline double getFar() const { return dist_far; }
      inline top10::math::Vector getDirection() const { return -e2; }
      inline top10::math::Vector getRight() const { return e0; }
      inline top10::math::Vector getUp() const { return e1; }
      inline top10::math::Vector getCenter() const { return center; }

      inline void setCenter(top10::math::Vector c) { center = c; }
      void setDirection(top10::math::Vector dir, top10::math::Vector up);
      inline void setRatio(double w_over_h) { ratio = w_over_h; }
      inline void setFOV(double degrees) { fov = M_PI*degrees/180.0; }
      inline void setNear(double n) { dist_near = n; update(); }
      inline void setFar(double f) { dist_far = f; update(); }

      //! Update the internal data
      /*! Must be called before using getDirection or getGLMatrix */
      void update();

      //! Return a transformation matrix usable to setup a GL camera.
      /*! Includes also the translation */
      void getGLMatrix(double out[4][4]) const;
      inline void getGLMatrix(double v[16]) const { getGLMatrix((double (*)[4])v); }

      top10::helpers::OverlapType overlap(const top10::math::AxisAlignedBox&) const;
      top10::helpers::OverlapType overlap(const top10::math::Box&) const;
      void drawGL() const;
      void setViewGL() const;

    private:
      top10::math::Vector center;
      //! e2 points backwards
      top10::math::Vector e0, e1, e2;
      double ratio;
      //! In rads
      double fov;
      double dist_near;
      double dist_far;

      top10::math::Plane left, right, upper, lower, far_plane, near_plane;
      top10::math::Vector nul,nur,nll,nlr, ful,fur,fll,flr;
      std::vector<top10::helpers::Polygon> polygons;
    };
  };

  namespace helpers {
    inline OverlapType GenericOctree__overlap(const top10::math::AxisAlignedBox& b, const top10::ui_interactive::Frustum& f)
    {
      return f.overlap(b);
    }
  };
};

#endif
