/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <SDL.h>
#include <GL/glu.h>
#include <plib/pu.h>

#include "util/PathFinder.hh"
#include "util/UserFile.hh"
#include "../Drawable.hh"
#include "Camera.hh"
#include "DialogManager.hh"
#include "MeshDialog.hh"
#include "FileMenuDialog.hh"
#include "StartingAreaDialog.hh"
#include "DecorationsDialog.hh"

using namespace std;
using top10::math::Vector;

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );      
  screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
  if (screen == NULL) {
    throw string("Could not set GL mode: ") + string(SDL_GetError());
  }

  // GL state init
  GLfloat light_position[] = {5.0, 10.0, 5.0, 1.0};
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);
  glClearColor(.4, .4, 1, 1);
  glMatrixMode(GL_MODELVIEW);
  glEnable(GL_LIGHT0);
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(90.0, ((GLdouble)w)/h, 0.2, 1000.0);

  return screen;
}

int main(int argc, char** argv)
{
  try {
    int screen_width = 800;
    int screen_height = 600;
    string filename;

    top10::util::PathFinder::addPath("data");
    top10::util::PathFinder::addPath("data/textures");
    top10::util::PathFinder::addPath("data/meshes");
    top10::util::PathFinder::addPath("data/tracks");
    top10::util::PathFinder::addPath("data/sounds");
    top10::util::PathFinder::addPath("data/karts");
    top10::util::PathFinder::addPath("data/fonts");
    top10::util::PathFinder::addPath(top10::util::UserFile::getPath());
    top10::util::UserFile::makePath(); // Just in case it does not exist.
    top10::util::PathFinder::addPath("data/defaults");

    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (string(argv[i]) == "-w") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_width = atoi(argv[i]);
      }
      else if (string(argv[i]) == "-h") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_height = atoi(argv[i]);
      }
      else {
	filename = argv[i];
      }
    }

    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw string("Couldn't initialize SDL: ") + string(SDL_GetError());

    SDL_Surface* screen = initGL(0 /* Depth not specified */, screen_width, screen_height, false /* No fullscreen */);
    SDL_EnableUNICODE(1);

    // PUI init
    puInit();
    puSetDefaultStyle(PUSTYLE_SMALL_SHADED);

    // Build the camera
    Camera camera;
    camera.setScreenSize(screen_width, screen_height);

    // Setup all the track editors
    top10::tracked::MainEditor main_ed;
    top10::tracked::MeshEditor mesh_ed;
    top10::tracked::SurfacesEditor surf_ed;
    top10::tracked::DecorationsEditor deco_ed;
    top10::tracked::CheckpointsEditor cps_ed;
    top10::tracked::StartingAreaEditor sa_ed;

    main_ed.setMeshEditor(&mesh_ed);
    main_ed.setSurfacesEditor(&surf_ed);
    main_ed.setDecorationsEditor(&deco_ed);
    main_ed.setCheckpointsEditor(&cps_ed);
    main_ed.setStartingAreaEditor(&sa_ed);
    surf_ed.setMeshEditor(&mesh_ed);
    deco_ed.setMeshEditor(&mesh_ed);
    cps_ed.setMeshEditor(&mesh_ed);
    sa_ed.setMeshEditor(&mesh_ed);
    mesh_ed.setCamera(&camera);

    // The drawables...
    std::vector<top10::tracked::Drawable*> drawables;
    drawables.push_back(&mesh_ed);
    drawables.push_back(&surf_ed);
    drawables.push_back(&deco_ed);
    drawables.push_back(&cps_ed);
    drawables.push_back(&sa_ed);

    // The dialog manager and the dialogs
    DialogManager dialog_manager(&main_ed, &mesh_ed, &surf_ed, &deco_ed, &cps_ed, &sa_ed);
    FileMenuDialog file_dialog(&dialog_manager);
    MeshDialog mesh_dialog(&dialog_manager);
    StartingAreaDialog sa_dialog(&dialog_manager);
    DecorationsDialog deco_dialog(&dialog_manager);

    // Main loop
    SDL_Event event;

    for (bool leave = false; !leave; ) {
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

      // Setup the view
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      camera.setViewGL();

      // Draw the grid
      for (int c = -500; c <= 500; c+=10) {
	glColor3f(0.0, 0.0, 0.2);
	glBegin(GL_LINES);
	glVertex3i(-500, 0, c);
	glVertex3i(500, 0, c);
	glVertex3i(c, 0, -500);
	glVertex3i(c, 0, 500);
	glEnd();
      }

      // Draw all drawable objects
      GLfloat u=0.0;
      for (std::vector<top10::tracked::Drawable*>::const_iterator it_drawable = drawables.begin();
	   it_drawable != drawables.end();
	   ++it_drawable) {
	glPolygonOffset(0.0, u);
	glEnable(GL_POLYGON_OFFSET_FILL);
	glEnable(GL_POLYGON_OFFSET_LINE);
	(*it_drawable)->drawGL();
	u = u-1.0;
      }
	
      // Restore the view
      glMatrixMode(GL_MODELVIEW);
      glPopMatrix();

      // Draw the PUI widgets
      puDisplay();

      SDL_GL_SwapBuffers();

      // Handle events
      SDL_WaitEvent(0);
      while (SDL_PollEvent(&event)) {
	switch (event.type) {
	case SDL_KEYDOWN:
	  if (event.key.keysym.sym == SDLK_ESCAPE) {
	    dialog_manager.toggleActive();
	  }
	  else if (dialog_manager.isActive()) {
	    // Forward event to pui
	    switch (event.key.keysym.sym) {
	    case SDLK_RIGHT: puKeyboard(PU_KEY_RIGHT, PU_DOWN); break;
	    case SDLK_LEFT:  puKeyboard(PU_KEY_LEFT, PU_DOWN); break;
	    default: 	     puKeyboard(event.key.keysym.unicode & 0x7f, PU_DOWN); break;
	    }
	  }
	  break;
	    
	case SDL_QUIT:
	  leave = true;
	  break;
	    
	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
	  if (dialog_manager.isActive()) {
	    // Forward to pui
	    int button=0;
	    if      (event.button.button == 1) button = PU_LEFT_BUTTON;
	    else if (event.button.button == 2) button = PU_MIDDLE_BUTTON;
	    else if (event.button.button == 3) button = PU_RIGHT_BUTTON;

	    int state=0;
	    if      (event.button.state == SDL_PRESSED)  state = PU_DOWN;
	    else if (event.button.state == SDL_RELEASED) state = PU_UP;

	    puMouse(button, state, event.button.x, event.button.y);
	  }
	  else camera.handleEvent(event);
	  break;

	case SDL_MOUSEMOTION:
	  if (!dialog_manager.isActive()) {
	    camera.handleEvent(event);
	  }
	  break;

	default:
	  break;
	}
      }
    }
  }
  catch (string& err) {
    cerr<<"Exception: "<<err<<endl;
    SDL_Quit();
    exit(1);
  }
  catch (...) {
    cerr<<"Unknown exception"<<endl;
    SDL_Quit();
    exit(1);
  }

  SDL_Quit();
  exit(0);
}
