/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACK_TRACK_HH
#define TOP10_TRACK_TRACK_HH

#include "math/Vertex.hh"
#include "math/AxisEnum.hh"
#include "util/Debug.hh"
#include "racing/StartingArea.hh"
#include "racing/CheckPoint.hh"
#include "helpers/PolygonSet.hh"
#include <set>
#include <string>

namespace top10 {

  namespace physX {
    class SurfacePatch;
  };

  namespace track {

    /**
       Interface of all tracks.
    */
    class Track: public top10::util::Debug
    {
    public:
      Track(const top10::helpers::PolygonSet& mesh, const std::set<std::string>& deco_mesh_names):
	friction_factor(1.0),
	polygons(mesh),
	decorative_meshes(deco_mesh_names),
	trimap(1, SetRow(1))
      {init();}

      //! Build a track with debugging enabled.
      Track(const top10::helpers::PolygonSet& mesh, const std::set<std::string>& deco_mesh_names,
	    const char* name, bool b):
	friction_factor(1.0),
	top10::util::Debug(name, b),
	polygons(mesh),
	decorative_meshes(deco_mesh_names),
	trimap(1, SetRow(1))
      {init();}

      void setStartingArea(top10::racing::StartingArea);
      void setCheckPoints(top10::racing::CheckPoints);

      inline top10::racing::StartingArea* getStartingArea() { return &starting_area; }
      inline top10::racing::CheckPoints* getCheckPoints() { return &checkpoints; }

      inline const top10::racing::StartingArea* getStartingArea() const { return &starting_area; }
      inline const top10::racing::CheckPoints* getCheckPoints() const { return &checkpoints; }

      inline const top10::helpers::PolygonSet* getMesh() const { return &polygons; }
      inline top10::helpers::PolygonSet* getMesh() { return &polygons; }

      std::set<std::string> getDecorations() const { return decorative_meshes; }

      /*!
	Compute the plane containing the surface patch situated under p
	\param p
	\param patch output parameter
	\return false if no surface patch could be found
      */
      bool getSurfacePatch(const top10::math::Vector& p, top10::physX::SurfacePatch& patch) const;

      /**
	 Build the structure that allows such a fast triangle retrievial ;)
      */
      void buildTriMap(int depth=5);

      //! Set the global friction factor
      /*! getSurfacePatch mulitplies the "stickyness" by this grip factor.
	Can be used to simulate wet or icy surfaces */
      void setGripFactor(double f) { friction_factor = f; }
      double getGripFactor() const { return friction_factor; }

    private:
      top10::racing::StartingArea starting_area;
      top10::racing::CheckPoints checkpoints;

      void init();

    public:
      //    private:
      typedef std::vector<top10::helpers::Polygon*> PolygonRefSet;
      typedef std::vector<PolygonRefSet> SetRow;
      typedef std::vector<SetRow> TriMap;
      
      const PolygonRefSet* debugTriMap(const top10::math::Vector&) const;

    private:
      double friction_factor;
      top10::helpers::PolygonSet polygons;
      std::set<std::string> decorative_meshes;

      /**
	 This is a 2d array. Every cell contains a list of triangles.
	 It allows to quick-test which triangle intersects with a given vertical
	 line. */
      TriMap trimap;

      void iterateBuild();
      static bool cellIntersect(const top10::helpers::Polygon&, double, double, double);
      static bool inPolygon(const top10::helpers::Polygon&, double, double);
      static void subdivide(const PolygonRefSet& triangles, double x, double z, double size,
			    PolygonRefSet& ll, PolygonRefSet& lr,
			    PolygonRefSet& ul, PolygonRefSet& ur);
      static bool diag1_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s);
      static bool diag2_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s);

      double cell_size;

      /// Coordinates of the bounding box
      top10::math::Vector min_v;
      top10::math::Vector max_v;
    };

    //! Load a new track, oriented and scaled correctly
    Track* loadTrack(std::string filename);

    //! Load the parts of a track (mesh, orientation...) without any correction
    void loadTrackParts(std::string filename,
			std::string* meshname, top10::math::AxisEnum *axis, bool* axis_neg, double* scale,
			top10::racing::StartingArea* area, top10::racing::CheckPoints* cps,
			std::set<std::string>* decorative_meshes,
			std::set<std::string>* road_surfaces,
			std::set<std::string>* dusty_surfaces,
			std::set<std::string>* border_surfaces,
			std::set<std::string>* dirt_surfaces,
			std::set<std::string>* grass_surfaces,
			std::set<std::string>* sand_surfaces);

    //! Counter-part of loadTrackParts.
    void saveMeshTrack(std::string filename,
		       std::string meshname, top10::math::AxisEnum axis, bool axis_neg, double scale,
		       top10::racing::StartingArea area, top10::racing::CheckPoints cps,
		       std::set<std::string> decorative_meshes,
		       std::set<std::string> road_surfaces,
		       std::set<std::string> dusty_surfaces,
		       std::set<std::string> border_surfaces,
		       std::set<std::string> dirt_surfaces,
		       std::set<std::string> grass_surfaces,
		       std::set<std::string> sand_surfaces);
  };
};

#endif
