/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Track.hh"

#include "helpers/3dsRead.hh"
#include "helpers/LwRead.hh"
#include "helpers/OrientMatrix.hh"
#include "physX/Surface.hh"
#include "physX/SurfacePatch.hh"
#include "util/PathFinder.hh"
#include "util/Range.hh"
#include "util/Parse.hh"

#include <fstream>

using namespace top10;
using namespace top10::track;

using top10::util::Error;
using top10::math::Vector;
using top10::helpers::Read3DS;
using top10::helpers::LWRead;
using top10::helpers::PolygonSet;
using top10::helpers::Polygon;
using top10::racing::CheckPoint;
using top10::racing::CheckPoints;
using top10::racing::StartingArea;

void Track::init()
{
  using top10::util::Range;
  if (polygons.empty()) throw Error("Triangle set empty in constructor");

  min_v = polygons[0].p[0];
  max_v = min_v;

  // Build the range of polygons making up the ground
  Range range(0, polygons.size()-1);
  for (std::set<std::string>::const_iterator deco = decorative_meshes.begin();
       deco != decorative_meshes.end();
       ++deco) {
    for (std::vector<top10::helpers::Mesh>::const_iterator mesh = polygons.meshes.begin();
	 mesh != polygons.meshes.end();
	 ++mesh) {
      if (mesh->name == *deco) range.remove(mesh->begin, mesh->end-1);
    }
  }

  for (Range::const_iterator idx=range.begin();
       idx!=range.end();
       ++idx) {
    top10::helpers::Polygon* poly = &(polygons[*idx]);

    for (int i=poly->p.size()-1; i>=0; --i) {
      if (poly->p[i].x < min_v.x) min_v.x = poly->p[i].x;
      if (poly->p[i].y < min_v.y) min_v.y = poly->p[i].y;
      if (poly->p[i].z < min_v.z) min_v.z = poly->p[i].z;

      if (poly->p[i].x > max_v.x) max_v.x = poly->p[i].x;
      if (poly->p[i].y > max_v.y) max_v.y = poly->p[i].y;
      if (poly->p[i].z > max_v.z) max_v.z = poly->p[i].z;
    }
    trimap[0][0].push_back(poly);
  }
  Vector _diag = max_v - min_v;

  // cell_size = max(_diax.x, _diag.y, _diag.z)
  cell_size = _diag.x;
  if (_diag.y > cell_size) cell_size = _diag.y;
  if (_diag.z > cell_size) cell_size = _diag.z;
}

void Track::buildTriMap(int depth)
{
  for (int i=0; i<depth; ++i) {
      iterateBuild();
  }
}

void Track::iterateBuild()
{
  // Save current values
  TriMap old_trimap = trimap;
  double old_cell_size = cell_size;
  int old_n = old_trimap.size();

  // Update new values
  trimap.clear();
  int n=old_n*2;
  trimap = TriMap(n, SetRow(n));
  cell_size /= 2.0;

  double x = min_v.x;
  for (int col=0;
       col < old_n;
       ++col, x+=old_cell_size) {
    double z=min_v.z;
    for (int row=0;
	 row < old_n;
	 ++row, z+=old_cell_size) {
      subdivide(old_trimap[col][row], x, z, cell_size,
		trimap[2*col][2*row], trimap[2*col+1][2*row],
		trimap[2*col][2*row+1], trimap[2*col+1][2*row+1]);
    }
  }
}

void Track::subdivide(const PolygonRefSet& polys, double x, double z, double size,
			  PolygonRefSet& ll, PolygonRefSet& lr,
			  PolygonRefSet& ul, PolygonRefSet& ur)
{
  for (PolygonRefSet::const_iterator poly=polys.begin();
       poly!=polys.end();
       ++poly) {
    if (cellIntersect(**poly, x, z, size)) {
       ll.push_back(*poly);
    }
    if (cellIntersect(**poly, x+size, z, size)) {
       lr.push_back(*poly);
    }
    if (cellIntersect(**poly, x, z+size, size)) {
       ul.push_back(*poly);
    }
    if (cellIntersect(**poly, x+size, z+size, size)) {
       ur.push_back(*poly);
    }
  }
}

static inline double cross(double x1, double y1, double x2, double y2)
{
	return x1*y2 - y1*x2;
}

bool Track::diag1_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s)
{
  double V1 = (x1 - X - (y1 -Y));
  double V2 = (x2 - X - (y2 - Y));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  V1 = cross(X-x1, Y-y1, x2-x1, y2-y1);
  V2 = V1 + s*((y2 - y1) - (x2 - x1));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  return true;	
}

bool Track::diag2_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s)
{
  double V1 = (-(x1 - X) - (y1 -Y));
  double V2 = (-(x2 - X) - (y2 - Y));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  V1 = cross(X-x1, Y-y1, x2-x1, y2-y1);
  V2 = V1 + s*((y2 - y1) + (x2 - x1));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  return true;	
}

bool Track::cellIntersect(const Polygon& polygon, double x, double z, double size)
{
  // Check if one of the vertices lies inside the square
  for (int i=polygon.p.size()-1; i>=0; --i) {
    if ((polygon.p[i].x >= x && polygon.p[i].x < x+size &&
	 polygon.p[i].z >= z && polygon.p[i].z < z+size)) {
      return true;
    }
  }
  
  // Check if one of the square vertices lies within the polygon
  if (inPolygon(polygon, x, z)) return true;
  if (inPolygon(polygon, x+size, z)) return true;
  if (inPolygon(polygon, x+size, z+size)) return true;
  if (inPolygon(polygon, x, z+size)) return true;
  
  // Check if polygon and square segments intersect
  // Trick: I test against the square's diagonals instead of its edges
  const int nseg = polygon.p.size();
  for (int i=nseg-1; i>=0; --i) {
    int i2 = (i==nseg-1)?0:(i+1);
    if (diag1_intersect(polygon.p[i].x, polygon.p[i].z, polygon.p[i2].x, polygon.p[i2].z, x, z, size)) return true;
    if (diag2_intersect(polygon.p[i].x, polygon.p[i].z, polygon.p[i2].x, polygon.p[i2].z, x, z+size, size)) return true;
  }
  
  return false;
}

bool Track::inPolygon(const Polygon& poly, double x, double z)
{

  double sign;
  /* to check if the point is inside a triangle, I use the cross product
     method.
  */
#define CROSS(a,b) ((poly.p[b].x - poly.p[a].x)*(z-poly.p[a].z) - (poly.p[b].z - poly.p[a].z)*(x-poly.p[a].x))

  sign = CROSS(0,1);
  // If the point is on the edge, I consider it is not in the polygon
  if (fabs(sign) < 1e-5) return false;

  const int nseg = poly.p.size();
  for (int i=0; i<nseg; ++i) {
    int i2 = (i==nseg-1)?0:(i+1);
    if ((CROSS(i,i2) * sign) < 0.0) return false;
  }
#undef CROSS

  return true;
}

bool Track::getSurfacePatch(const Vector& p_orig, top10::physX::SurfacePatch& out)
  const
{
  int found = 0;

  // p_orig expressed in local base
  Vector p( p_orig.x - min_v.x,
	    0,
	    p_orig.z - min_v.z);
  int c = int(p.x/cell_size);
  int l = int(p.z/cell_size);
  Vector p1, p2, p3;

  if (c >= trimap.size()) return false;
  if (c < 0) return false;

  if (l >= trimap[c].size()) return false;
  if (l < 0) return false;

  for (PolygonRefSet::const_iterator ptri = trimap[c][l].begin();
       ptri != trimap[c][l].end();
       ptri++) {
    if (inPolygon(**ptri, p_orig.x, p_orig.z)) {
      p1 = (*ptri)->p[0];
      p2 = (*ptri)->p[1];
      p3 = (*ptri)->p[2];
      // Check if normal points upward
      if (((p2-p1)^(p3-p1)).y < 0) {
	Vector temp = p2;
	p2 = p3;
	p3 = temp;
      }

      out = top10::physX::SurfacePatch(top10::math::Plane(p1, p2, p3),
				       polygons.surfaces.at((*ptri)->surface_name).phys_surface);
      out.grip *= friction_factor;
      found++;
      return true;
    }
  }

  if (found > 1) {
    std::cerr<<"Found more than one triangle: "<<found<<std::endl;
  }

  return found > 0;
}

const Track::PolygonRefSet* Track::debugTriMap(const Vector& p_orig) const
{
  // p_orig expressed in local base
  Vector p( p_orig.x - min_v.x,
	    0,
	    p_orig.z - min_v.z);
  int c = int(p.x/cell_size);
  int l = int(p.z/cell_size);

  if (c >= trimap.size()) return 0;
  if (c < 0) return 0;

  if (l >= trimap[c].size()) return 0;
  if (l < 0) return 0;

  return &trimap[c][l];
}

void Track::setStartingArea(StartingArea sa) { starting_area = sa; }
void Track::setCheckPoints(CheckPoints cp) { checkpoints = cp; }

//if partial=true, we already parsed the opening < before coming here
static Vector parseVector(std::istream& in, bool partial=false)
{
  std::string token;

  if (!partial) {
    in >> token;
    if (token != "<") throw Error("Expected <, got ") + token;
  }

  double x, y, z;
  in >> x >> y >> z;

  Vector v(x, y, z);

  in >> token;
  if (token != ">") throw Error("Expected >, got ") + token;

  return v;
}

static void writeVector(std::ostream& out, Vector v)
{
  out << " < ";
  for (int i=0; i<3; ++i) out<<v[i]<<" ";
  out << " > ";
}

Track* top10::track::loadTrack(std::string filename)
{
  using namespace top10::helpers;

  Track* track;

  std::string meshname;
  top10::math::AxisEnum axis;
  bool axis_neg;
  double scale;
  top10::racing::StartingArea area;
  top10::racing::CheckPoints cps;
  std::set<std::string> decos;
  std::set<std::string> road_surfaces;
  std::set<std::string> dusty_surfaces;
  std::set<std::string> border_surfaces;
  std::set<std::string> dirt_surfaces;
  std::set<std::string> grass_surfaces;
  std::set<std::string> sand_surfaces;

  loadTrackParts(filename, &meshname, &axis, &axis_neg, &scale, &area, &cps, &decos,
		 &road_surfaces, &dusty_surfaces, &border_surfaces,
		 &dirt_surfaces, &grass_surfaces, &sand_surfaces);

  std::string mesh_path = top10::util::PathFinder::find(meshname);
  if (mesh_path.empty()) throw Error("Could not find ")+meshname;
  PolygonSet* mesh = PolygonSet::loadNew(mesh_path);
  mesh->transform(makeOrientation(axis, axis_neg, scale));
  for (std::vector<top10::helpers::Surface>::iterator surf = mesh->surfaces.begin();
       surf != mesh->surfaces.end();
       ++surf) {
    if (road_surfaces.find(surf->name) != road_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceConcrete();

    else if (dusty_surfaces.find(surf->name) != dusty_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceDust();

    else if (border_surfaces.find(surf->name) != border_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceBorder();

    else if (dirt_surfaces.find(surf->name) != dirt_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceDirt();

    else if (grass_surfaces.find(surf->name) != grass_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceGrass();

    else if (sand_surfaces.find(surf->name) != sand_surfaces.end())
      surf->phys_surface = top10::physX::SurfaceSand();
  }
  track = new Track(*mesh, decos);
  delete mesh;
  
  track->setStartingArea(area);
  track->setCheckPoints(cps);

  // Build the mapping from squares to triangles to accelerate collision detection
  track->buildTriMap();

  return track;
}

void top10::track::loadTrackParts(std::string filename,
				  std::string* meshname, top10::math::AxisEnum *axis,
				  bool* axis_neg, double* scale,
				  StartingArea* area, CheckPoints* cps,
				  std::set<std::string>* decorative_meshes,
				  std::set<std::string>* road_surfaces,
				  std::set<std::string>* dusty_surfaces,
				  std::set<std::string>* border_surfaces,
				  std::set<std::string>* dirt_surfaces,
				  std::set<std::string>* grass_surfaces,
				  std::set<std::string>* sand_surfaces)
{
  using top10::util::parseString;

  // We don't modify the output args directly, for atomicity.
  std::string tmp_meshname;
  top10::math::AxisEnum tmp_axis;
  bool tmp_axis_neg;
  double tmp_scale;
  top10::racing::StartingArea tmp_area;
  top10::racing::CheckPoints tmp_cps;
  std::list<top10::helpers::SurfaceFilter*> tmp_filters;
  std::set<std::string> tmp_decorative_meshes;
  std::set<std::string> tmp_road_surfaces;
  std::set<std::string> tmp_dusty_surfaces;
  std::set<std::string> tmp_border_surfaces;
  std::set<std::string> tmp_dirt_surfaces;
  std::set<std::string> tmp_grass_surfaces;
  std::set<std::string> tmp_sand_surfaces;

  std::string file_path = top10::util::PathFinder::find(filename);
  if (file_path.empty()) throw Error("Could not find ")+filename;
  std::ifstream in(file_path.c_str());
  if (!in) throw Error(std::string("Could not open ") + file_path);
  
  /* Expected:
     MESH mesh_filename
     UP <X, Y, Z, -X, -Y or -Z>
     SCALE 1.2345

  */
  std::string token;
  in >> token;

  if (token == "MESH") {
    tmp_meshname = parseString(in);
    // We only want to keep the basename of the file, not the path.
    std::string::size_type pos = tmp_meshname.rfind("/", tmp_meshname.size());
    if (pos != std::string::npos) {
      tmp_meshname = tmp_meshname.substr(pos+1);
    }

    in >> token;
    if (token != "UP") throw Error(std::string("Expected UP, got ") + token);
    in >> token;
    top10::math::Matrix4 trans;

    tmp_axis_neg = false;
    if (token == "X")       { tmp_axis = top10::math::X; tmp_axis_neg = false; }
    else if (token == "-X") { tmp_axis = top10::math::X; tmp_axis_neg = true;  }
    else if (token == "Y")  { tmp_axis = top10::math::Y; tmp_axis_neg = false; }
    else if (token == "-Y") { tmp_axis = top10::math::Y; tmp_axis_neg = true;  }
    else if (token == "Z")  { tmp_axis = top10::math::Z; tmp_axis_neg = false; }
    else if (token == "-Z") { tmp_axis = top10::math::Z; tmp_axis_neg = true;  }
    else throw Error(std::string("Expected X, Y or Z, got ") + token);

    in >> token;
    if (token != "SCALE") throw Error(std::string("Expected SCALE, got ") + token);
    in >> tmp_scale;
  }
  else throw Error(std::string("Expected MESH or VECTOR, got: ") + token);

  /* Read the checkpoints. Grammar:
     BEGIN
     checkpoints
     END
     
     checkpoints := empty | checkpoint checkpoints
     checkpoint := vector vector
     vector := <x y z>
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    in >> token;
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");
    
    Vector p1 = parseVector(in, true);
    Vector p2 = parseVector(in);
    
    tmp_cps.push_back(CheckPoint(p1, p2));
  }
  
  /* Read the starting area. Expected:
     vector vector vector.
     (point1 point2 direction)
  */
  Vector p1 = parseVector(in);
  Vector p2 = parseVector(in);
  Vector dir = parseVector(in);
  
  tmp_area = StartingArea(p1, p2, dir);

  /* Read the surface properties
     BEGIN
     material_name surface_type
     ...
     END

     surface_type is: ROAD, DUSTY_ROAD, BORDER, DIRT, GRASS, SAND
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    token = parseString(in);
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    std::string material_name = token;
    in>>token;
    if (token == "ROAD") tmp_road_surfaces.insert(material_name);
    else if (token == "DUSTY_ROAD") tmp_dusty_surfaces.insert(material_name);
    else if (token == "BORDER") tmp_border_surfaces.insert(material_name);
    else if (token == "DIRT") tmp_dirt_surfaces.insert(material_name);
    else if (token == "GRASS") tmp_grass_surfaces.insert(material_name);
    else if (token == "SAND") tmp_sand_surfaces.insert(material_name);
    else throw Error("Expected surface type, got ") + token;
  }

  /* Read the names of meshes that are not part of the ground. Expected:
     BEGIN
     mesh name
     ...
     END
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    token = parseString(in);

    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    tmp_decorative_meshes.insert(token);
  }

  // Everything went well, modify the output args now.
  *meshname = tmp_meshname;
  *axis = tmp_axis;
  *axis_neg = tmp_axis_neg;
  *scale = tmp_scale;
  *area = tmp_area;
  *cps = tmp_cps;
  *decorative_meshes = tmp_decorative_meshes;
  *road_surfaces = tmp_road_surfaces;
  *dusty_surfaces = tmp_dusty_surfaces;
  *border_surfaces = tmp_border_surfaces;
  *dirt_surfaces = tmp_dirt_surfaces;
  *grass_surfaces = tmp_grass_surfaces;
  *sand_surfaces = tmp_sand_surfaces;
}

void top10::track::saveMeshTrack(std::string filename,
				 std::string meshname, top10::math::AxisEnum axis, bool axis_neg, double scale,
				 StartingArea area, CheckPoints cps,
				 std::set<std::string> decorative_meshes,
				 std::set<std::string> road_surfaces,
				 std::set<std::string> dusty_surfaces,
				 std::set<std::string> border_surfaces,
				 std::set<std::string> dirt_surfaces,
				 std::set<std::string> grass_surfaces,
				 std::set<std::string> sand_surfaces)
{
  std::ofstream out(filename.c_str());
  if (!out) throw Error("Could not open ") + filename;
  
  /* Expected:
     MESH <mesh file type: LW or 3DS> mesh_filename
     UP <X, Y, Z, -X, -Y or -Z>
     SCALE 1.2345

  */
  // We only want to keep the basename of the file, not the path.
  std::string::size_type pos = meshname.rfind("/", meshname.size());
  if (pos != std::string::npos) {
    meshname = meshname.substr(pos+1);
  }
  out << "MESH " << meshname << std::endl;

  out << "UP " ;
  if (axis_neg) out<<"-";

  if (axis >=0 && axis < 3) {
    const char names[3] = {'X', 'Y', 'Z'};
    out << names[(int)axis];
  }
  else throw Error("Internal Error: invalid axis index");

  out<<std::endl;

  out << "SCALE " << scale << std::endl;

  /* Write the checkpoints. Grammar:
     BEGIN
     checkpoints
     END
     
     checkpoints := empty | checkpoint checkpoints
     checkpoint := vector vector
     vector := <x y z>
  */
  out << "BEGIN" << std::endl;
  for (CheckPoints::const_iterator cp = cps.begin();
       cp != cps.end();
       ++cp) {
    writeVector(out, cp->getPoint1());
    writeVector(out, cp->getPoint2());
  }
  out<< "END" << std::endl;

  /* Write the starting area. Expected:
     vector vector vector
     (point1 point2 direction)
  */

  writeVector(out, area.getPoint1());
  writeVector(out, area.getPoint2());
  writeVector(out, area.getDirection());

  /* Write the surface filters.
     BEGIN
     material_name surface_type
     ...
     END

     surface_type is: ROAD, DUSTY_ROAD, BORDER, DIRT, GRASS, SAND
  */
  out << "BEGIN" << std::endl;

  for (std::set<std::string>::const_iterator str = road_surfaces.begin();
       str != road_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" ROAD"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = dusty_surfaces.begin();
       str != dusty_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" DUSTY_ROAD"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = dirt_surfaces.begin();
       str != dirt_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" DIRT"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = border_surfaces.begin();
       str != border_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" BORDER"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = grass_surfaces.begin();
       str != grass_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" GRASS"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = sand_surfaces.begin();
       str != sand_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" SAND"<<std::endl;
  }

  out << "END" << std::endl;

  /* Write the names of decorative meshes (those that are not part of the ground) */
  out<<"BEGIN"<<std::endl;
  for (std::set<std::string>::const_iterator it = decorative_meshes.begin();
       it != decorative_meshes.end();
       ++it) {
    out<<"\""<<(*it)<<"\""<<std::endl;
  }
  out<<"END"<<std::endl;
}
