/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Wheel.hh"
#include "Point.hh"
#include "track/Track.hh"
#include "math/Matrix.hh"

using top10::physX::Wheel;
using top10::track::Track;
using top10::physX::Point;
using top10::math::Vector;
using top10::math::OrthoNorm3;

#define SMALL_EPS 1e-8
#define BIG_EPS 1e-4

const double tyre_mass = 1.0; // 1Kg
const int dt = 10; // 10ms

Track* make_track()
{
  top10::helpers::Surface surface;
  top10::helpers::PolygonSet one_flat_polygon;
  one_flat_polygon.surf_names.push_back("Surface");
  one_flat_polygon.surfaces.push_back(surface);
  top10::helpers::Polygon poly;
  poly.p.push_back(Vector(-1000, 0, -1000));
  poly.p.push_back(Vector(-1000, 0, 1000));
  poly.p.push_back(Vector(1000, 0, 1000));
  poly.p.push_back(Vector(1000, 0, -1000));
  one_flat_polygon.push_back(poly);
  
  Track* track = new Track(one_flat_polygon);
  track->buildTriMap(1);
  return track;
}

Wheel* make_wheel(Track* track)
{
  Wheel* wheel = new Wheel(track, 1.0 /*long_k*/, 1.0 /*side_k*/,
			   0.3 /*width*/, 1.0 /*radius*/, 0.2/*inertia*/);
  OrthoNorm3 I3;
  wheel->setTransform(I3, Vector(0,1,0));
  return wheel;
}

int main(int argc, char** argv)
{
  Track* track = make_track();
  Wheel* wheel;
  Point center;

  // **************************************** TEST: bouncing
  std::cout<<"TEST: bouncing"<<std::endl;
  wheel = make_wheel(track);
  center = Point(0,1,0,tyre_mass);
  wheel->setOrigin(Vector(0, 2, 0));
  center.pos = Vector(0,2,0);
  wheel->setAngle(0.0);
  wheel->setDriveTorque(0.0);
  for (int t=0; t<10000; t+=dt) {
    wheel->applyForce(Vector(0, -tyre_mass*9.81, 0));
    wheel->collideTrack(center.speed, dt);
    if (wheel->getGroundCollision() != Wheel::None) {
      center.speed += wheel->getImpulse();
      if (wheel->getGroundCollision() == Wheel::Collision)
	std::cout<<"REPORT: collision at "<<(t/1000.0)<<" "<<center.speed<<std::endl;;
    }
    wheel->computeForces(center.speed, dt/1000.0);
    Vector force = wheel->getForce();
    center.accel = force/center.mass;
    center.integrate(dt/1000.0);
    center.pos += Vector(0, wheel->getHeight(), 0);
    wheel->setHeight(0.0);
    center.shift();
    wheel->setOrigin(center.pos);
  }
  double speed_size = center.speed.size();
  if (speed_size > 0.1) std::cout<<"FAILED: wheel should be resting "<<center.speed<<std::endl;
  else std::cout<<"OK: Wheel is resting"<<center.speed<<std::endl;

  // ************************************ TEST: rolling forward
  std::cout<<"TEST: rolling forward"<<std::endl;
  delete wheel;
  wheel = make_wheel(track);
  center = Point(0,1,0, tyre_mass);
  wheel->setAngle(0.0);
  wheel->setDriveTorque(1.0);
  for (int t=0; t<10000; t+=dt) {
    wheel->applyForce(Vector(0, -tyre_mass*9.81, 0));
    wheel->collideTrack(center.speed, dt);
    if (wheel->getGroundCollision() != Wheel::Contact) {
      std::cout<<"FAILED: collision type != contact"<<std::endl;
      break;
    }
    wheel->computeForces(center.speed, dt/1000.0);
    Vector force = wheel->getForce();
    center.accel = force/center.mass;
    center.integrate(dt/1000.0);
    center.shift();
    wheel->setOrigin(center.pos);
  }
  speed_size = center.speed.size();
  if (speed_size < SMALL_EPS) std::cout<<"FAILED: speed too small: "<<speed_size<<std::endl;
  else std::cout<<"OK: speed big enough: "<<speed_size<<std::endl;
  Vector speed_norm = center.speed / speed_size;
  Vector speed_err = speed_norm - Vector(1, 0, 0);
  double err = speed_err.size();
  if (err > BIG_EPS)
    std::cout<<"FAILED: speed vector misoriented "<<speed_err<<", "<<(err*100.0)<<"%"<<std::endl;
  else
    std::cout<<"OK: speed vector"<<speed_err<<", "<<(err*100.0)<<"%"<<std::endl;


  // ************************************ TEST: turning
  std::cout<<"TEST: turning"<<std::endl;
  delete wheel;
  wheel = make_wheel(track);
  center = Point(0,1,0, tyre_mass);
  wheel->setAngle(0.1);
  wheel->setDriveTorque(1.0);
  for (int t=0; t<10000; t+=dt) {
    wheel->applyForce(Vector(0, -tyre_mass*9.81, 0));
    wheel->collideTrack(center.speed, dt);
    if (wheel->getGroundCollision() != Wheel::Contact) {
      std::cout<<"FAILED: collision type != contact"<<std::endl;
      break;
    }
    wheel->computeForces(center.speed, dt/1000.0);
    Vector force = wheel->getForce();
    center.accel = force/center.mass;
    center.integrate(dt/1000.0);
    center.shift();
    wheel->setOrigin(center.pos);
  }
  if (center.pos.z > SMALL_EPS || center.pos.x < SMALL_EPS)
    std::cout<<"FAILED: Did not turn to the left: "<<center.pos<<std::endl;
  else std::cout<<"OK: Turned to the left: "<<center.pos<<std::endl;

  return 0;
}
