/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "RigidBody.hh"
#include "Point.hh"

#define BIG_EPS 1e-5
#define SMALL_EPS 1e-9

using top10::physX::RigidBody;
using top10::physX::Point;
using top10::math::Vector;

RigidBody* makeBody()
{
  RigidBody* body = new RigidBody;
  std::vector<Point> pts;

  pts.push_back(Point(Vector(1,1,0), 1));
  pts.push_back(Point(Vector(-1,1,0), 1));
  pts.push_back(Point(Vector(-1,-1,0), 1));
  pts.push_back(Point(Vector(1, -1, 0), 1));
  body->setMasses(pts);
  return body;
}

int main(int argc, char** argv)
{
  RigidBody* body = makeBody();

  // Check if the center of mass lies in (0,0,0)
  std::cout<<"TEST: center of mass"<<std::endl;
  Vector center_err = body->getMassCenter();
  if (center_err.size() > BIG_EPS) std::cout<<"FAILED: center_err = "<<center_err<<std::endl;
  else std::cout<<"OK: center_err"<<std::endl;

  // Apply two forces, check if the body responds as expected
  std::cout<<"TEST: downward force"<<std::endl;
  int dt=10;
  for (int t=0; t<10000; t+=dt) {
    body->applyForceAtL(Vector(-1,0,0), Vector(0, -10, 0));
    body->applyForceAtL(Vector(1,0,0), Vector(0, -10, 0));
    body->integrate(dt/1000.0);
  }
  double expected_y = -20.0*100.0/8.0;
  center_err = body->getMassCenter() - Vector(0, expected_y, 0);
  double err = fabs(center_err.y/expected_y);
  if (err > 0.01)
    std::cout<<"FAILED: center_err = "<<center_err<<", "<<(100.0*err)<<"%"<<std::endl;
  else std::cout<<"OK: center_err "<<(100.0*err)<<"%"<<std::endl;
  
  // Body should not have rotated
  Vector speed_ext, speed_center;
  speed_ext = body->getSpeedAtL(Vector(-1,0,0));
  speed_center = body->getSpeedAtL(Vector(0, 0, 0));
  if ((speed_ext - speed_center).size() > BIG_EPS)
    std::cout<<"FAILED: speeds "<<speed_ext<<" != "<<speed_center<<std::endl;
  else std::cout<<"OK: speeds"<<std::endl;

  if (body->getAngularSpeedL().size() > BIG_EPS)
    std::cout<<"FAILED: angular speed "<<body->getAngularSpeedL()<<std::endl;
  else std::cout<<"OK: angular speed"<<std::endl;

  std::cout<<"TEST: non-central force"<<std::endl;
  delete body;
  body = makeBody();
  // Apply one non-central force
  for (int t=0; t<10000; t+=dt) {
    body->applyForceAtL(Vector(-1,0,0), body->getOrientation()*Vector(0, -10, 0));
    body->integrate(dt/1000.0);
  }
  Vector w_speed = body->getAngularSpeedL();
  if (w_speed.size() < SMALL_EPS) std::cout<<"FAILED: w_speed (size) = "<<w_speed<<std::endl;
  else std::cout<<"OK: w_speed large enough = "<<w_speed<<std::endl;
  w_speed /= w_speed.size();
  if ((w_speed - Vector(0,0,1)).size() > BIG_EPS)
    std::cout<<"FAILED: w_speed (direction) = "<<w_speed<<std::endl;
  else std::cout<<"OK: w_speed well oriented"<<std::endl;

  // Apply opposite force
  std::cout<<"TEST: opposite non-central force"<<std::endl;
  for (int t=0; t<10000; t+=dt) {
    body->applyForceAtL(Vector(-1,0,0), body->getOrientation() * Vector(0, 10, 0));
    body->integrate(dt/1000.0);
  }
  w_speed = body->getAngularSpeedL();
  if (w_speed.size() > BIG_EPS) std::cout<<"FAILED: w_speed = "<<w_speed<<std::endl;
  else std::cout<<"OK: w_speed canceled"<<std::endl;

  delete body;
  return 0;
}
