/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <fstream>
#include <cmath>

#include "Kart.hh"
#include "util/error.hh"
#include "track/Track.hh"
#include "Options.hh"
#include "math/Triangle.hh"
#include "math/intersections.hh"

using namespace top10::physX;
using namespace std;
using namespace top10::math;

double top10::physX::Kart::STEER_ANGLE;
double top10::physX::Kart::STEER_RADIUS;
double top10::physX::Kart::TYRE_MASS;
double top10::physX::Kart::TYRE_WIDTH;
double top10::physX::Kart::DRIVER_MASS;
double top10::physX::Kart::KART_MASS;
double top10::physX::Kart::FRONT_X;
double top10::physX::Kart::FRONT_Y;
double top10::physX::Kart::FRONT_Z;
double top10::physX::Kart::BACK_X;
double top10::physX::Kart::BACK_Y;
double top10::physX::Kart::BACK_Z;
double top10::physX::Kart::FRONT_INERTIA;
double top10::physX::Kart::BACK_INERTIA;
double top10::physX::Kart::ENGINE_INERTIA;
double top10::physX::Kart::FRONT_RADIUS;
double top10::physX::Kart::BACK_RADIUS;
double top10::physX::Kart::FRONT_LONG_K;
double top10::physX::Kart::BACK_LONG_K;
double top10::physX::Kart::FRONT_SIDE_K;
double top10::physX::Kart::BACK_SIDE_K;
double top10::physX::Kart::FRONT_STIFFNESS;
double top10::physX::Kart::BACK_STIFFNESS;
double top10::physX::Kart::FRONT_DAMPING;
double top10::physX::Kart::BACK_DAMPING;
double top10::physX::Kart::FRONT_MAX_DAMPING_FORCE;
double top10::physX::Kart::BACK_MAX_DAMPING_FORCE;
double top10::physX::Kart::ENGINE_RPS_MAX;
double top10::physX::Kart::ENGINE_RPS_OPT;
double top10::physX::Kart::ENGINE_FT_TORQUE0;
double top10::physX::Kart::ENGINE_FT_TORQUE1;
double top10::physX::Kart::ENGINE_NT_TORQUE1;
double top10::physX::Kart::ENGINE_NT_TORQUE2;
double top10::physX::Kart::BRAKE_K;
double top10::physX::Kart::COLLISION_FACTOR = 0.1;

void Kart::initDefaultParameters(istream& in)
{
  string keyword;
  while (!in.eof()) {
    in>>keyword;
    if (keyword.size() == 0 || in.eof()) break;

    // Check if it is a comment
    if (keyword.c_str()[0] == '#') {
      // Skip line
      in.ignore(256, '\n');
    }
    else if (keyword == "maximum_steer_angle") {
      double w; in>>w;
      STEER_ANGLE = w;
    }
    else if (keyword == "steer_radius") {
      double w; in>>w;
      STEER_RADIUS = w;
    }
    else if (keyword == "tyre_mass") {
      double w; in>>w;
      TYRE_MASS = w;
    }
    else if (keyword == "tyre_width") {
      double w; in>>w;
      TYRE_WIDTH = w;
    }
    else if (keyword == "kart_mass") {
      double w; in>>w;
      KART_MASS = w;
    }
    else if (keyword == "driver_mass") {
      double w; in>>w;
      DRIVER_MASS = w;
    }
    else if (keyword == "wheel_front_inertia") {
      double i; in>>i;
      FRONT_INERTIA = i;
    }
    else if (keyword == "wheel_back_inertia") {
      double i; in>>i;
      BACK_INERTIA = i;
    }
    else if (keyword == "engine_inertia") {
      double i; in>>i;
      ENGINE_INERTIA = i;
    }
    else if (keyword == "width") {
      double w; in>>w;
      FRONT_Z = w/2;
      BACK_Z = w/2;
    }
    else if (keyword == "length") {
      double l; in>>l;
      FRONT_X = l/2;
      BACK_X = -l/2;
    }
    else if (keyword == "front_radius") {
      double i; in>>i;
      FRONT_RADIUS = i;
    }
    else if (keyword == "back_radius") {
      double i; in>>i;
      BACK_RADIUS = i;
    }
    else if (keyword == "front_long_k") {
      in>>FRONT_LONG_K;
    }
    else if (keyword == "back_long_k") {
      in>>BACK_LONG_K;
    }
    else if (keyword == "front_side_k") {
      in>>FRONT_SIDE_K;
    }
    else if (keyword == "back_side_k") {
      in>>BACK_SIDE_K;
    }
    else if (keyword == "front_stiffness") {
      in>>FRONT_STIFFNESS;
    }
    else if (keyword == "back_stiffness") {
      in>>BACK_STIFFNESS;
    }
    else if (keyword == "front_damping") {
      in>>FRONT_DAMPING;
    }
    else if (keyword == "back_damping") {
      in>>BACK_DAMPING;
    }
    else if (keyword == "front_max_damping_force") {
      in>>FRONT_MAX_DAMPING_FORCE;
    }
    else if (keyword == "back_max_damping_force") {
      in>>BACK_MAX_DAMPING_FORCE;
    }
    else if (keyword == "engine_rps_max") {
      in>>ENGINE_RPS_MAX;
    }
    else if (keyword == "engine_rps_opt") {
      in>>ENGINE_RPS_OPT;
    }
    else if (keyword == "engine_ft_torque0") {
      in>>ENGINE_FT_TORQUE0;
    }
    else if (keyword == "engine_ft_torque1") {
      in>>ENGINE_FT_TORQUE1;
    }
    else if (keyword == "engine_nt_torque1") {
      in>>ENGINE_NT_TORQUE1;
    }
    else if (keyword == "engine_nt_torque2") {
      in>>ENGINE_NT_TORQUE2;
    }
    else if (keyword == "brake_k") {
      in>>BRAKE_K;
    }
    else if (keyword == "collision_factor") {
      in>>COLLISION_FACTOR;
    }
    else {
      throw top10::util::Error("Kart parse: parse error: " + keyword);
    }
  }
}

Kart::Kart(top10::track::Track* track, Vector pos, Vector e1):
  fl(FRONT_X, FRONT_Y, -FRONT_Z),
  fr(FRONT_X, FRONT_Y, FRONT_Z),
  rl(BACK_X, BACK_Y, -BACK_Z),
  rr(BACK_X, BACK_Y, BACK_Z),
  top((BACK_X*3+FRONT_X)/4, 0.5, 0),
  wheel_fl(track, FRONT_LONG_K, FRONT_SIDE_K, FRONT_RADIUS,
	   "Front Left Wheel", debugKartWheels),
  wheel_fr(track, FRONT_LONG_K, FRONT_SIDE_K, FRONT_RADIUS,
	   "Front Right Wheel", debugKartWheels),
  wheel_rl(track, BACK_LONG_K, BACK_SIDE_K, BACK_RADIUS,
	   "Back Left Wheel", debugKartWheels),
  wheel_rr(track, BACK_LONG_K, BACK_SIDE_K, BACK_RADIUS,
	   "Back Right Wheel", debugKartWheels),
  engine(&wheel_rl, &wheel_rr, ENGINE_RPS_MAX, ENGINE_RPS_OPT,
	 ENGINE_FT_TORQUE0, ENGINE_FT_TORQUE1,
	 ENGINE_NT_TORQUE1, ENGINE_NT_TORQUE2,
         BRAKE_K),
  body("Kart body", debugKartBodies),
  is_colliding(false)
{
  // Find the altitude a the starting point
  SurfacePatch start_patch;
  if (!track->getSurfacePatch(pos, start_patch)) {
    throw top10::util::Error("Starting position not over track");
  }
  double h = (pos * start_patch.normal) - start_patch.d;

  /* Rotate body according to its intended orientation */
  /* Transformation matrix:
      x0 0 -x2
      0  1 0
      x2 0 x0 */
  double e1_size = e1.size();
  if (e1_size < SMALL_VECTOR) e1 = Vector(1,0,0);
  else e1 /= e1_size;

  Vector vs[3] = {Vector(e1.x, 0, e1.z),
		 Vector(0, 1, 0),
		 Vector(-e1.z, 0, e1.x)};
  OrthoNorm3 orient; orient = Matrix3(vs);

  std::vector<Point> points;
  points.push_back(Point(fl, TYRE_MASS+KART_MASS/4));
  points.push_back(Point(fr, TYRE_MASS+KART_MASS/4));
  points.push_back(Point(rl, TYRE_MASS+KART_MASS/4));
  points.push_back(Point(rr, TYRE_MASS+KART_MASS/4));
  points.push_back(Point(top, DRIVER_MASS));
  body.setMasses(points);
  body.setTranslation(pos - Vector(0, h-0.3, 0));
  body.setOrientation(orient);

  wheel_fl.setTransform(orient, body.getGlobalFromLocal(fl));
  wheel_fr.setTransform(orient, body.getGlobalFromLocal(fr));
  wheel_rl.setTransform(orient, body.getGlobalFromLocal(rl));
  wheel_rr.setTransform(orient, body.getGlobalFromLocal(rr));

  // Compute the bounding box, used for collisions with other objects
  Vector center = (fl + fr + rl + rr)*0.25;
  double s1 = TYRE_WIDTH + ((FRONT_Z>BACK_Z)?FRONT_Z:BACK_Z);
  double s2 = (FRONT_RADIUS+BACK_RADIUS+FRONT_X-BACK_X)/2.0;
  double s3 = (FRONT_RADIUS>BACK_RADIUS)?FRONT_RADIUS:BACK_RADIUS;
  original_bbox = Box(center, Vector(0,0,s1), Vector(s2,0,0), Vector(0,s3,0));

  // Boxes used for collision response
  fl_box = AxisAlignedBox(fl, FRONT_RADIUS, FRONT_RADIUS, TYRE_WIDTH);
  fr_box = AxisAlignedBox(fr, FRONT_RADIUS, FRONT_RADIUS, TYRE_WIDTH);
  rl_box = AxisAlignedBox(rl, BACK_RADIUS, BACK_RADIUS, TYRE_WIDTH);
  rr_box = AxisAlignedBox(rr, BACK_RADIUS, BACK_RADIUS, TYRE_WIDTH);
  front_box = AxisAlignedBox(Vector(FRONT_X,FRONT_Y,0), 0.1, FRONT_RADIUS, FRONT_Z);
  back_box = AxisAlignedBox(Vector(BACK_X,BACK_Y,0), 0.1, BACK_RADIUS, BACK_Z);
  left_box = AxisAlignedBox(Vector(0.5*(FRONT_X+BACK_X),FRONT_Y,-BACK_Z), 0.5*(FRONT_X-BACK_X),
			    FRONT_RADIUS ,0.1);
  right_box = AxisAlignedBox(Vector(0.5*(FRONT_X+BACK_X),FRONT_Y,BACK_Z), 0.5*(FRONT_X-BACK_X),
			     FRONT_RADIUS, 0.1);
}

void Kart::getRegistered(World& w)
{
  w.objects.push_back(this);
}

void Kart::steer(double steer_frac)
{
  steer_angle = STEER_ANGLE * steer_frac;

  // Angle of the inside wheel
  double alpha = steer_angle;

  // Angle of the outside wheel
  double beta = atan2(STEER_RADIUS*sin(alpha),
		      STEER_RADIUS*cos(alpha) + 2*FRONT_Z);

  if (steer_frac > 0) {
    wheel_fl.setAngle(alpha);
    wheel_fr.setAngle(beta);
  }
  else {
    wheel_fr.setAngle(alpha);
    wheel_fl.setAngle(beta);
  }
}

void Kart::accel(double x)
{
  engine.accelerate(x);
}

void Kart::brake(double x)
{
  engine.brake(x);
}


double Kart::getRadPerSec() const
{
  double w = engine.getAngularVelocity();
  return w>0?w:0.0;
}

double Kart::getMaxRadPerSec() const
{
  return ENGINE_RPS_MAX;
}

double Kart::getBumps() const
{
  double ret = (wheel_fl.getBumpFactor() + wheel_fr.getBumpFactor()
          + wheel_rl.getBumpFactor() + wheel_rr.getBumpFactor())/4.0;

  std::cout<<"Kart::getBumps "<<ret<<std::endl;
  return ret;
}

top10::math::Box Kart::getBoundingBox() const
{
  top10::math::Box bbox = original_bbox;
  bbox.translate(-body.getMassCenterL());
  bbox.rotate(body.getOrientation());
  bbox.translate(body.getMassCenterL());
  bbox.translate(body.getTranslation());
  return bbox;
}

void Kart::collideTrack(double dt)
{
  body.clearForces();
  wheel_fl.clearForces();
  wheel_fr.clearForces();
  wheel_rl.clearForces();
  wheel_rr.clearForces();

  Vector up = body.getOrientation()*Vector(0,1,0);

  wheel_fl.collideTrack(body.getSpeedAtL(fl), dt);
  wheel_fr.collideTrack(body.getSpeedAtL(fr), dt);
  wheel_rl.collideTrack(body.getSpeedAtL(rl), dt);
  wheel_rr.collideTrack(body.getSpeedAtL(rr), dt);
}

void Kart::collideObjects(const Octree::ShapeVec* polyset, const Octree::ShapeRefs& polygons, double dt)
{
  using top10::helpers::PolygonSet;
  using top10::helpers::Polygon;

  // Only those polygons that actually collide with the bounding box
  std::vector<Polygon> polygons2(polygons.size()); 
  Box bbox = getBoundingBox();
  Matrix3 M = body.getOrientation().transpose();
  for (Octree::ShapeRefs::const_iterator ref = polygons.begin();
       ref != polygons.end();
       ++ref) {
    const top10::helpers::Polygon& poly(polyset->at(*ref));
    if (poly.p.size() >= 3) {
      bool overlap = false;
      for (int i=2; !overlap && i<poly.p.size(); ++i) {
	top10::math::Triangle triangle;
	triangle.p[0] = poly.p[0];
	triangle.p[i-1] = poly.p[i-1];
	triangle.p[i] = poly.p[i];
	
	overlap = top10::math::intersect(bbox, triangle);
      }

      if (overlap) {
	Polygon local_poly = poly;
	local_poly.translate(-body.getMassCenter());
	local_poly.rotate(M);
	polygons2.push_back(local_poly);
      }
    }
  }

  // Collision response: find out what parts of the kart collided

  double normals[8][3] = {
    {1,0,-1}, {1,0,1}, {-1,0,-1}, {-1,0,1},
    {1,0,0}, {-1,0,0}, {0,0,-1}, {0,0,1}
  };
  Vector pts[8] = {
    fl,fr,rl,rr, fl_box.getCenter(), fr_box.getCenter(), rl_box.getCenter(), rr_box.getCenter()
  };
  AxisAlignedBox* boxes[8] = {
    &fl_box, &fr_box, &rl_box, &rr_box, &front_box, &back_box, &left_box, &right_box
  };

  // Linear + angular impulses
  for (int i=0; i<4; ++i) {
    Vector speed = M*body.getSpeedAtL(pts[i]);
    Vector normal(normals[i][0], normals[i][1], normals[i][2]);
    if (speed*normal > 0) {
      if (top10::helpers::intersect(polygons2, *boxes[i])) {
	Vector impulse = computeImpulse(speed, normal, pts[i]);
	body.applyImpulseAtL(pts[i], impulse);
	is_colliding = true;
	//	std::cerr<<"Collision: "<<i<<" impulse "<<impulse<< " speed "<<speed<<std::endl;
	return;
      }
    }
  }
  // Linear-only impulses
  for (int i=4; i<8; ++i) {
     Vector speed = M*body.getSpeedAtL(pts[i]);
     Vector normal(normals[i][0], normals[i][1], normals[i][2]);
     if (speed*normal > 0) {
       if (top10::helpers::intersect(polygons2, *boxes[i])) {
	 Vector impulse = computeLinearImpulse(speed, normal);
	 body.applyLinearImpulseL(impulse);
	 is_colliding = true;
	 //	 std::cerr<<"Collision: "<<i<<" impulse "<<impulse<< " speed "<<speed<<std::endl;
	 return;
       }
     }
  }
}

Vector Kart::computeLinearImpulse(Vector speed, Vector normal) const
{
  //TODO: collision factor should depend on the object with which this kart is colliding
  return -normal*((normal*speed)*(COLLISION_FACTOR+1.0)*body.getMass());
}

Vector Kart::computeImpulse(Vector speed, Vector normal, Vector pos) const
{
  Vector r = pos - body.getMassCenterL();
  return -normal*((normal*speed)*(COLLISION_FACTOR+1.0)/(1.0/body.getMass() + normal*(((r^normal)*body.getInertiaInv())^r)));
}

void Kart::update(double G, double dt)
{
  engine.update(dt);

  if (!is_colliding) {
    Vector up = body.getOrientation()*Vector(0,1,0);

    // Forces from the springs
    double damping = up*wheel_fl.getImpulse()*FRONT_DAMPING;
    if (damping > FRONT_MAX_DAMPING_FORCE) damping = FRONT_MAX_DAMPING_FORCE;
    else if (damping < -FRONT_MAX_DAMPING_FORCE) damping = -FRONT_MAX_DAMPING_FORCE;
    Vector force_fl = -(wheel_fl.getHeight()*FRONT_STIFFNESS + damping)*up;

    damping = up*wheel_fr.getImpulse()*FRONT_DAMPING;
    if (damping > FRONT_MAX_DAMPING_FORCE) damping = FRONT_MAX_DAMPING_FORCE;
    else if (damping < -FRONT_MAX_DAMPING_FORCE) damping = -FRONT_MAX_DAMPING_FORCE;
    Vector force_fr = -(wheel_fr.getHeight()*FRONT_STIFFNESS + damping)*up;

    damping = up*wheel_rl.getImpulse()*BACK_DAMPING;
    if (damping > BACK_MAX_DAMPING_FORCE) damping = BACK_MAX_DAMPING_FORCE;
    else if (damping < -BACK_MAX_DAMPING_FORCE) damping = -BACK_MAX_DAMPING_FORCE;
    Vector force_rl = -(wheel_rl.getHeight()*BACK_STIFFNESS + damping)*up;

    damping = up*wheel_rr.getImpulse()*BACK_DAMPING;
    if (damping > BACK_MAX_DAMPING_FORCE) damping = BACK_MAX_DAMPING_FORCE;
    else if (damping < -BACK_MAX_DAMPING_FORCE) damping = -BACK_MAX_DAMPING_FORCE;
    Vector force_rr = -(wheel_rr.getHeight()*BACK_STIFFNESS + damping)*up;

    wheel_fl.applyForce(force_fl);
    wheel_fr.applyForce(force_fr);
    wheel_rl.applyForce(force_rl);
    wheel_rr.applyForce(force_rr);

    // Compute the reaction from the ground
    wheel_fl.computeForces(body.getSpeedAtL(fl), dt);
    wheel_fr.computeForces(body.getSpeedAtL(fr), dt);
    wheel_rl.computeForces(body.getSpeedAtL(rl), dt);
    wheel_rr.computeForces(body.getSpeedAtL(rr), dt);

    // Apply forces to the body
    body.applyForceAt(body.getMassCenter(), Vector(0,-body.getMass()*G,0));
    body.applyForceAtL(fl, -force_fl + projectV(Plane(up, 0), wheel_fl.getForce()));
    body.applyForceAtL(fr, -force_fr + projectV(Plane(up, 0), wheel_fr.getForce()));
    body.applyForceAtL(rl, -force_rl + projectV(Plane(up, 0), wheel_rl.getForce()));
    body.applyForceAtL(rr, -force_rr + projectV(Plane(up, 0), wheel_rr.getForce()));
  } else {
    body.moveBack();
  }
  is_colliding = false;
}

void Kart::integrate(double dt)
{
  // Update the body
  body.integrate(dt);

  // Update the wheels
  wheel_fl.setTransform(body.getOrientation(), body.getGlobalFromLocal(fl));
  wheel_fr.setTransform(body.getOrientation(), body.getGlobalFromLocal(fr));
  wheel_rl.setTransform(body.getOrientation(), body.getGlobalFromLocal(rl));
  wheel_rr.setTransform(body.getOrientation(), body.getGlobalFromLocal(rr));

  // Update wheel angular velocities + engine
  wheel_fl.setAngularSpeed(wheel_fl.getAngularSpeed() + dt*wheel_fl.getTorque()/FRONT_INERTIA);
  wheel_fr.setAngularSpeed(wheel_fr.getAngularSpeed() + dt*wheel_fr.getTorque()/FRONT_INERTIA);
  double w_accel = wheel_rl.getTorque() + wheel_rr.getTorque() + engine.getTorque();
  w_accel /= 2*BACK_INERTIA + ENGINE_INERTIA;
  double w_speed = engine.getAngularVelocity() + dt*w_accel;
  wheel_rl.setAngularSpeed(w_speed);
  wheel_rr.setAngularSpeed(w_speed);
  engine.setAngularVelocity(w_speed);
}
