/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_PLANE_H
#define TOP10_PLANE_H

#include "util/Dumpable.hh"
#include "Vertex.hh"
#include "Matrix.hh"

namespace top10 {
  namespace math {

    class Plane: public top10::util::Dumpable
    {
    public:
      /**
	 Default plane: the ground (horizontal, altitude 0)
      */
      Plane(): normal(0, 1, 0), d(0) {}

      /**
	 Plane defined by three points
      */
      Plane(Vector p1, Vector p2, Vector p3);

      /**
	 Plane defined by a point and a normal vector.
	 normal.size() must be 1 !!!
      */
      Plane(Vector p, Vector normal);

      Plane(Vector _normal, double _d): normal(_normal), d(_d) {}

      void translate(Vector);
      void rotate(const Matrix3&);

      inline bool isBelow(Vector p) const { return p*normal >= d; }

      void dumpTo(std::ostream&) const;
      void loadFrom(std::istream&);

      //FIXME:
      //    private:

      /// Normal to the plane. Must have length = 1
      Vector normal;
  
      /// Distance from the center
      double d;
    };

    /**
       Compute intersection between a segment and a plane.
       \param plane The plane
       \param A First end point of the segment
       \param B Second end point of the segment
       \param result Intersection point, if any
  
       \return True if intersection, else false
    */
    bool intersect(const Plane& plane, const Vector& A, const Vector& B, Vector& result);

    /**
       Compute intersection between a ray and a plane.
       \param plane The plane
       \param A end point of the segment
       \param V direction of the ray
       \param result Intersection point, if any
       \param distance from A to the intersection point, if any

       \return True if intersection, else false
    */
    bool intersectRay(const Plane& plane, const Vector& A, const Vector& V, Vector& result, double& d);

    /**
       Project a vertex on a plane.
       The resulting vertex belongs to the plane
    */
    Vector project(const Plane& plane, const Vector& v);

    /**
       Project a vector on a plane
    */
    Vector projectV(const Plane& plane, const Vector& v);

  };
};

#endif
