#include "3dsRead.hh"

#include <lib3ds/file.h>                        
#include <lib3ds/mesh.h>
#include <lib3ds/node.h>
#include <lib3ds/material.h>
#include <lib3ds/matrix.h>
#include <lib3ds/vector.h>

#include <sstream>

using namespace top10;
using namespace top10::helpers;

using top10::util::Error;
using top10::math::Vector;
using top10::math::Matrix4;
using top10::math::Translation4;

Read3DS::Read3DS(const char* filename)
{
  Lib3dsFile* file = lib3ds_file_load(filename);
  if (!file) throw Error(std::string("Could not open ")+filename);
  lib3ds_file_eval(file, 0);

  try {
    top10::math::Identity4 trans;
    int polygon_index = 0;
    for (Lib3dsNode* node = file->nodes; node; node = node->next) {
      loadNode(file, node, trans);
      // Include all newly created polygons into the new mesh
      int polygon_index_end = size();
      mesh_names.push_back(node->name);
      Mesh new_mesh;
      new_mesh.begin = polygon_index;
      new_mesh.end = polygon_index_end;
      new_mesh.name = node->name;
      new_mesh.hidden = false;
      polygon_index = polygon_index_end;
      meshes.push_back(new_mesh);
    }
  }
  catch (Error err) {
    lib3ds_file_free(file);
    throw err;
  }

  lib3ds_file_free(file);
}

void Read3DS::loadNode(Lib3dsFile* file, Lib3dsNode* node, Matrix4 trans)
{
  // Handle children if any
  for (Lib3dsNode* child = node->childs; child; child = child->next)
    loadNode(file, node, trans);

  // Handle this mesh
  Lib3dsMesh* mesh=lib3ds_file_mesh_by_name(file, node->name);

  // Magical transformations. 3DS manages transformation oddly, IMHO
  Lib3dsMatrix M;
  lib3ds_matrix_copy(M, mesh->matrix);
  lib3ds_matrix_inv(M);
  Matrix4 M2(M); M2 = M2.transpose();
  Matrix4 nodeM(node->matrix); nodeM = nodeM.transpose();
  Lib3dsObjectData* obj_data = &node->data.object;
  Matrix4 nodeT = Translation4(-Vector(obj_data->pivot)); nodeT = nodeT.transpose();
  trans = trans*nodeM*nodeT*M2;

  // For each face...
  for (unsigned int p=0; p<mesh->faces; ++p) {
    Polygon poly;

    Lib3dsFace *f=&mesh->faceL[p];
    Lib3dsMaterial *mat=0;

    // Retrieve the material (surface)
    if (f->material[0]) {
      mat=lib3ds_file_material_by_name(file, f->material);
    }

    if (mat) {
      // Check if we already loaded it...
      std::vector<std::string>::const_iterator surf_f = std::find(surf_names.begin(), surf_names.end(), std::string(f->material));
      int surf_idx = 0;

      // ... no we did not
      if (surf_f == surf_names.end()) {
	surf_names.push_back(std::string(f->material));
	Surface new_surface;
	new_surface.name = f->material;
	new_surface.phys_surface = top10::physX::SurfaceSlip();
	new_surface.r = (unsigned char)(mat->diffuse[0]*255);
	new_surface.g = (unsigned char)(mat->diffuse[1]*255);
	new_surface.b = (unsigned char)(mat->diffuse[2]*255);
	new_surface.a = (unsigned char)((1.0 - mat->transparency)*255);
	if (mat->texture1_map.name[0]) {
	  new_surface.texture_filename = std::string(mat->texture1_map.name);
	  std::cerr<<"Texture name: "<<mat->texture1_map.name<<std::endl;
	}
	surfaces.push_back(new_surface);

	surf_idx = surf_names.size()-1;
      }

      // ... yes we did
      else {
	// Convert the iterator into an array index.
	for (std::vector<std::string>::const_iterator p = surf_names.begin();
	     p != surf_f;
	     ++p, ++surf_idx);
      }
      poly.surface_name = surf_idx;
    }
    else throw std::string("Missing material info");

    for (int i=0; i<3; ++i) {
      int idx = f->points[i];
      Vector v(mesh->pointL[idx].pos[0],
	       mesh->pointL[idx].pos[1],
	       mesh->pointL[idx].pos[2]);

      v = trans*v;
      poly.p.push_back(v);

      if (mesh->texels != 0) {
	if (mesh->texels != mesh->points) {
	  std::ostringstream buf;
	  buf<<"Bad number of texels "<<mesh->texels<<", should be: "<<mesh->points<<std::ends;
	  throw buf.str();
	}

	TexCoord texel = {mesh->texelL[idx][0], -mesh->texelL[idx][1]};
	poly.texels.push_back(texel);
      }
      else {
	TexCoord dummy = {0,0};
	poly.texels.push_back(dummy);
      }
    }

    push_back(poly);
  }
}
