#ifndef TOP10_UTILREFCOUNT_HH
#define TOP10_UTILREFCOUNT_HH

#include <cassert>
#include <vector>

namespace top10 {
namespace util {

class RefCount {
public:
  RefCount();
  RefCount(const RefCount&);
  
  void use() const;
  bool release() const;

#ifndef NDEBUG
  int getCount() const { return ref_count; }
  unsigned long getSalt() const { return salt;}
  ~RefCount();
#endif

private:
  mutable int ref_count;
#ifndef NDEBUG
  mutable bool first_use;
  unsigned long salt;

#endif
};

template<typename T>
class Ref {
public:
  explicit Ref(T* ptr): ptr(ptr) { if (ptr) ptr->use(); }
  Ref(): ptr(0) {}
  Ref(const Ref<T>& other): ptr(other.getPtr()) { if (ptr) ptr->use(); }

  template<typename T2>
    Ref<T>& operator=(const Ref<T2>& other) {
    check();
    if (ptr == other.ptr) return *this;
    if (ptr && ptr->release()) destroy();
    ptr = other.getPtr();
    if (ptr) ptr->use();
    check();
    return *this;
  }

  Ref<T>& operator=(T* optr) {
    check();
    if (optr == ptr) return *this;
    if (ptr && ptr->release()) destroy();
    ptr = optr;
    if (ptr) ptr->use();
    check();
    return *this;
  }
  
  template<typename T2>
    Ref<T2> dyncast() const { check(); return Ref<T2>(dynamic_cast<T2*>(ptr)); }
  
  bool isValid() const { check(); return ptr != 0; }

  T* getPtr() const { check(); return ptr; }
    
  T& operator*() const { check(); return *ptr; }
  T* operator->() const { check(); return ptr; }
  bool operator==(const Ref<T>& other) const { return ptr == other.ptr; }
  bool operator!=(const Ref<T>& other) const { return ptr != other.ptr; }
  bool operator<(const Ref<T>& other) const { return ptr < other.ptr; }

  void discard() { check(); if (ptr && ptr->release()) destroy(); ptr=0; }
  
  ~Ref() { check(); if (ptr && ptr->release()) destroy(); }

private:
  inline void check() const {
#ifdef NDEBUG
    if (ptr == 0) return;
    assert(ptr->getCount() > 0);
    assert(ptr->getSalt() == 0xdeadaed);
#endif
  }

  inline void destroy() const {
#if 1
    delete(ptr);
#else
    ptr->T::~T();
#endif
  }

  T* ptr;
};

//! A dynamic array of references to T
/*! Using std::vector< Ref<T> > just does not seem to work, don't know why ?!
  This implementation actually uses pointers to the objects, not refs.
*/
template<typename T>
class RefArray: private std::vector<T*>
{
  typedef typename std::vector<T*> Super;
  
public:
  friend class Cell;
  friend class ConstCell;
  
  //! Behaves like const Ref<T>&
  class ConstCell
  {
  public:
    ConstCell(const RefArray* array, int idx): array(array), idx(idx) {}

    inline operator T*()   { return array->at(idx); }
    inline T* operator->() { return array->at(idx); }
    inline T& operator*()  { return *array->at(idx); }    
    
  private:
    const RefArray* array;
    int idx;
  };
  
  //! Behaves like Ref<T>&
  class Cell
  {
  public:
    Cell(RefArray* array, int idx): array(array), idx(idx) {}
    
    inline Cell& operator=(T* r) {
      if (r) r->use();
      if (array->at(idx)) array->at(idx)->release();
      array->at(idx) = r;
      return *this;
    }
    
    inline operator T*()   { return array->at(idx); }
    inline T* operator->() { return array->at(idx); }
    inline T& operator*()  { return *array->at(idx); }
    
  private:
    RefArray* array;
    int idx;
  };
  
  //! Empty array of refs
  RefArray() {}
  //! Array of n null refs
  RefArray(int n): std::vector<T*>(n) {
    for(int i=0; i<n; ++i) at(i) = 0; 
  }
  //! Copy an array of refs
  /*! Increases all ref counts  */
  RefArray(const RefArray<T>& other): std::vector<T*>(other) {
    for (int i=0; i<size(); ++i) if (at(i)) at(i)->use();
  }
  
  //! Copy operator
  /*! Increases all ref counts  */
  RefArray& operator=(const RefArray<T>& other) {
    std::vector<T*>::operator=(other);
    for (int i=0; i<size(); ++i) if (at(i)) at(i)->use();
  }
  
  //! Decrease all ref counts, and deallocate if necessary
  ~RefArray() {
    for (int i=0; i<size(); ++i) if (at(i)) if (at(i)->release()) delete at(i);   
  }
  
  //! Access element at position idx
  /*! Will push back null refs up to idx if idx >= size */
  Cell operator[](int idx) {
    for (int i=size(); i<=idx; ++i) push_back(0);
    return Cell(this, idx);
  }
  
  ConstCell operator[](int idx) const {
    return ConstCell(this, idx);
  }
  
  inline int size() const { return Super::size(); }
  inline bool empty() const { return Super::empty(); }
  inline void push_back(T* r) { Super::push_back(r); if (r) r->use(); }
  inline void remove(T* r) {
    for (int i=0; i<size(); ) {
      if (at(i) == r) {
        if (r && r->release())
          delete r;
        Super::erase(begin() + i);
      }
      else ++i;
    } 
  }
  inline void erase(int i) {
    if (i>=0 && i<size()) {
      if (at(i) && at(i)->release())
        delete at(i);
      Super::erase(begin() +i);
    }
  }
  inline void clear() {
    while (!empty()) erase(0);   
  }
  
};

}
}

#endif
