/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_DUMPABLE_HH
#define TOP10_DUMPABLE_HH

#include <iostream>

#include "error.hh"

namespace top10 {
  namespace util {

    /**
       Interface for all objects that can be saved and restored
    */
    class Dumpable {
    public:
      DECL_ERROR(ParseError);

    public:
      /**
	 Save the object's state to a stream.
      */
      virtual void dumpTo(std::ostream&) const =0;

      /**
	 Restore a state from a stream.
	 \exception ParseError The state read is not valid
      */
      virtual void loadFrom(std::istream&) =0;

      virtual ~Dumpable() {}

      /**
	 Helper function to write a data of any type to a stream.
	 It ouputs the raw binary description. Better fitted for basic types.
      */
      template<typename T>
      static inline void output(std::ostream& out, const T& d) {
	out.write((char*)&d, sizeof(d));
      }

      /**
	 Helper function to read a data of any type from a stream.
      */
      template<typename T>
      static inline void input(std::istream& in, T& d) {
	in.read((char*)&d, sizeof(d));
      }

    };

  };
};

#endif
