/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Triangle.hh"
#include "SDL.h"
#include "util/PathFinder.hh"
#include "physX/Octree.hh"
#include <GL/glu.h>
#include <list>
#include "math/Box.hh"

using namespace std;
using top10::math::Vector;

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );      
  screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
  if (screen == NULL) {
    throw string("Could not set GL mode: ") + string(SDL_GetError());
  }

  return screen;
}

static inline void glVertex(Vector v)
{
  glVertex3f(v.x, v.y, v.z);
}

int main(int argc, char** argv)
{
  try {
    int screen_width = 800;
    int screen_height = 600;
    string filename;

    top10::util::PathFinder::addPath("data");
    top10::util::PathFinder::addPath("data/textures");

    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (string(argv[i]) == "-w") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_width = atoi(argv[i]);
      }
      else if (string(argv[i]) == "-h") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_height = atoi(argv[i]);
      }
      else {
	filename = argv[i];
      }
    }

    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw string("Couldn't initialize SDL: ") + string(SDL_GetError());

    SDL_Surface* screen = initGL(0 /* Depth not specified */, screen_width, screen_height, false /* No fullscreen */);

    // GL state init
    GLfloat light_position[] = {5.0, 10.0, 5.0, 1.0};
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);
    glClearColor(.4, .4, 1, 1);
    glMatrixMode(GL_MODELVIEW);
    glEnable(GL_LIGHT0);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    glViewport(0, 0, screen_width, screen_height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(90.0, ((GLdouble)screen_width)/screen_height, 0.2, 1000.0);

    // Load the mesh
    top10::helpers::PolygonSet* mesh = top10::helpers::PolygonSet::loadNew(filename);
    top10::ui_interactive::TriangleSet triangles(mesh);
    top10::physX::Octree octree(mesh);
    std::list<top10::math::AxisAlignedBox> blocks = octree.getAllBlocks();

    // Main loop
    SDL_Event event;
    double xangle = 0.0;
    double yangle = 0.0;
    top10::math::Matrix4 M = top10::math::Identity4();
    top10::math::Vector viewer_pos(0, 0, -10);

    for (bool leave = false; !leave; ) {
      // Update view
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      GLdouble ref[4][4];
      top10::math::Matrix4 M2(M.transpose());
      M2.toGL(ref);
      glMatrixMode(GL_MODELVIEW);
      glLoadMatrixd((GLdouble*)ref);
      glTranslated(-viewer_pos.x, -viewer_pos.y, -viewer_pos.z);

      // Draw the orizontal plane
      glDisable(GL_LIGHTING);
      glColor3f(0, 0, .3);
      glBegin(GL_LINES);
      for (int x = -100; x <= 100; x += 10) {
	glVertex3i(x, 0, -100);
	glVertex3i(x, 0, 100);

	glVertex3i(-100, 0, x);
	glVertex3i(100, 0, x);
      }
      glEnd();

      // Draw the object
      //      glEnable(GL_LIGHTING);
      triangles.drawGL();

      // Draw the blocks in the octree
      glColor3f(1, 0, 0);
      for (std::list<top10::math::AxisAlignedBox>::const_iterator block = blocks.begin();
	   block != blocks.end();
	   ++block) {
	Vector vertices[8];
	block->getVertices(vertices);

	glBegin(GL_LINE_STRIP);
	glVertex(vertices[0]);
	glVertex(vertices[1]);
	glVertex(vertices[3]);
	glVertex(vertices[2]);
	glVertex(vertices[6]);
	glVertex(vertices[4]);
	glVertex(vertices[5]);
	glVertex(vertices[7]);
	glVertex(vertices[3]);
	glEnd();

	glBegin(GL_LINES);
	glVertex(vertices[0]);
	glVertex(vertices[2]);

	glVertex(vertices[6]);
	glVertex(vertices[7]);

	glVertex(vertices[1]);
	glVertex(vertices[5]);

	glVertex(vertices[0]);
	glVertex(vertices[4]);
	glEnd();
      }

      SDL_GL_SwapBuffers();

      // Handle events
      while (SDL_PollEvent(&event)) {
	switch (event.type) {
      	case SDL_KEYDOWN:
	  if (event.key.keysym.sym == SDLK_ESCAPE) leave=true;

	  else if (event.key.keysym.sym == SDLK_SPACE) {
	    for (int i=0; i<4; ++i) {
	      for (int j=0; j<4; ++j) {
		std::cerr<<" "<<M(i,j);
	      }
	      std::cerr<<std::endl;
	    }
	    std::cerr<<std::endl;
	  }

	  break;

      	case SDL_QUIT:
      	  leave = true;
          break;

      	case SDL_MOUSEMOTION:
	  if (event.motion.state == 1) {
	    xangle += 1.6 * event.motion.xrel / screen_width;
	    yangle += 1.6 * event.motion.yrel / screen_height;

	    top10::math::Vector e[3];
	    e[0] = top10::math::Vector(cos(xangle), 0, sin(xangle));
	    e[2] = e[0] ^ top10::math::Vector(0, 1, 0);
	    e[2] = top10::math::Rotation3(-yangle, e[0]) * e[2];
	    e[1] = e[2] ^ e[0];
	    M = top10::math::Matrix3(e);
	  }
	  if (event.motion.state > 1) {
	    top10::math::Vector e1 = M.getColumn(0);
	    top10::math::Vector e2 = M.getColumn(2);
	    viewer_pos += -50.0*e1*event.motion.xrel/screen_width +50.0*e2*event.motion.yrel/screen_height;
	  }

	  break;
      		
      	default:
      	  break;
      	}
      }
    }

    // Exit cleanup
    delete mesh;
  }
  catch (string& err) {
    cerr<<"Exception: "<<err<<endl;
    SDL_Quit();
    exit(1);
  }
  catch (...) {
    cerr<<"Unknown exception"<<endl;
    SDL_Quit();
    exit(1);
  }

  SDL_Quit();
  exit(0);
}

#include "helpers/GenericOctree-template.cpp"
