/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

//C++ stuff
#include <iostream>
#include <fstream>

//SDL Stuff
#include <SDL/SDL.h>

//My stuff
#include "Simulation.hh"
#include "Options.hh"
#include "audio.hh"
#include "UIPanel.hh"
#include "MainMenu.hh"
#include "Controls.hh"
#include "util/PathFinder.hh"
#include "util/UserFile.hh"
#include "util/SlowTime.hh"
#include "util/GlobalOptions.hh"

using namespace top10::ui_interactive;
using namespace std;

top10::track::Track* parseNewTrack(std::string);

static void close_joy(int, void* joy)
{
  SDL_JoystickClose((SDL_Joystick*)joy);
}

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int w, int h, bool full_screen)
{
  using top10::util::Error;

  SDL_Surface* screen;
  int bits_per_channel(5);
  int depth = -1;
  
  // Try 32 bpp
  if (SDL_VideoModeOK( w, h, 32, SDL_OPENGL | (full_screen?SDL_FULLSCREEN:0))) {
    depth = 32;
    bits_per_channel = 8;
  }    
  // Try 24 bpp
  else if (SDL_VideoModeOK( w, h, 24, SDL_OPENGL | (full_screen?SDL_FULLSCREEN:0))) {
    depth = 24;
    bits_per_channel = 8;
  }
  // Try 16 bpp
  else if (SDL_VideoModeOK( w, h, 16, SDL_OPENGL | (full_screen?SDL_FULLSCREEN:0))) {
    depth = 16;
    bits_per_channel = 5;
  }
  else depth = -1;

  if (depth != -1) {
    cout<<"depth = "<<depth<<" bpc = "<<bits_per_channel<<endl;
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16);
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
    SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, 8 );
    screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
    if (screen == NULL) {
      throw Error(string("Could not set GL mode: ") + string(SDL_GetError()));
    }
  }
  else {
    throw Error("No true color video mode available.");
  }

  return screen;
}

/// Deinitialize SDL when destroyed
class SDL_Destroyer
{
public:
  static const int quit =1;
  static const int sound =2;
  static const int joystick =4;

public:
  SDL_Destroyer(): do_quit(false), do_joy(false), do_sound(false) {};

  /// Tell what parts of the sdl to deinit
  void enable(int what)
  {
    if (what & quit) do_quit = true;
    if (what & sound) do_sound = true;
    if (what & joystick) do_joy = true;
  }

  ~SDL_Destroyer()
  {
    // BUG: SDL deinitializers crash !
    if (do_sound) {
      SDL_CloseAudio();
    }
    //    if (do_joy) cerr<<"TODO: close joy"<<endl;
    if (do_quit) SDL_Quit();
  }

private:
  bool do_quit;
  bool do_joy;
  bool do_sound;
};


int main(int argc, char** argv)
{
  using namespace top10::ui_interactive::options;
  using top10::util::Error;

  try {
    SDL_Destroyer sdl_closer;  // Close sdl when destroyed
    SDL_Surface* screen;
    top10::util::PathFinder path_finder = top10::util::PathFinder::defaultPathFinder();

    // Load options
    try {
      top10::util::GlobalOptions::getGlobalOptions()->load("options.xml");
    }
    catch (top10::util::Error& e) {
      std::cerr<<"Error while reading options: "<<e<<std::endl;
    }
    
    parse(argc, argv);
    top10::util::global_time.setSlowFactor(slow_factor);

    /* Initialize the SDL library */
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK | SDL_INIT_AUDIO) < 0 ) {
      throw Error(string("Couldn't initialize SDL: ") + string(SDL_GetError()));
    }
    
    // Open all joysticks
    int n_sticks = SDL_NumJoysticks();
    std::vector<SDL_Joystick*> joysticks;
    for (int i=0; i<n_sticks; ++i) {
      SDL_Joystick* joy = SDL_JoystickOpen(i);
      joysticks.push_back(joy);
      if (joy) {
	std::cout<<"Opened joystick "<<i<<std::endl;
	std::cout<<"name: "<<SDL_JoystickName(i)<<" ";
	std::cout<<"Axes: "<<SDL_JoystickNumAxes(joy)<<" ";
	std::cout<<"Buttons: "<<SDL_JoystickNumButtons(joy)<<" ";
	std::cout<<"Balls: "<<SDL_JoystickNumBalls(joy)<<" ";
      }
      else std::cout<<"Failed top open joystick "<<i<<std::endl;
    }

    /* Clean up on exit, exit on window close and interrupt */
    sdl_closer.enable(SDL_Destroyer::quit);
  
    // Initialize GL stuff
    screen = initGL(w, h, full_screen);
    top10::graphX::Renderer render;
    render.initGL(w,h);
    
    // First load the default settings
    std::string kart_path = path_finder.find("default/default.physx");
    if (kart_path.empty()) throw Error("Could not find default physics settings");
    else {
      std::ifstream kart_in(kart_path.c_str());
      if (!kart_in) throw Error("Could not open ")+kart_path;
      top10::physX::Kart::initDefaultParameters(kart_in);
    }
    // Then load the specific settings
    if (!kart_filename.empty()) {
      kart_path = path_finder.find(kart_filename + "/" + kart_filename + ".physx");
      if (kart_path.empty()) throw Error("Could not find ") + kart_filename + " physics settings";
      else {
	std::ifstream kart_in(kart_path.c_str());
	if (!kart_in) throw Error("Could not open ")+kart_path;
	top10::physX::Kart::initDefaultParameters(kart_in);
      }
    }

    // Init font lib
    fntInit();

    // Audio
    AudioDevice* audio_device(0);
    if (use_audio) {
      audio_device = new AudioDevice;

      /* Open the audio device, forcing the desired format */
      SDL_AudioSpec wanted;
      wanted.freq = 22050;
      wanted.format = AUDIO_S16;
      wanted.channels = 2;    /* 1 = mono, 2 = stereo */
      wanted.samples = 256;  /* Good low-latency value for callback */
      wanted.callback = fill_audio;
      wanted.userdata = (void*)audio_device;
      
      if ( SDL_OpenAudio(&wanted, NULL) < 0 ) {
	throw Error(string("Couldn't open audio: ") + string(SDL_GetError()));
      }
      sdl_closer.enable(SDL_Destroyer::sound);
      cout<<"Audio device opened"<<endl;
    }
    
    // Load keyboard mapping
    ActionMapping actions;
    try {
      ActionMapping tmp_actions;
      if (!path_finder.find("inputs").empty())
	tmp_actions = loadActionMapping("inputs");
      actions = tmp_actions;
    }
    catch (std::string err) {
      std::cerr<<"Could not load action mapping: "<<err<<std::endl;
    }
    // When using the keyboard, steer events should not steer 100%
    int steer_amount = (int)(getOptD("Control.Steer.Factor")*32767);
    if (steer_amount < 0 || steer_amount >32767) {
      std::cout<<"Steer amount "<<steer_amount<<" out of range. Forcing in-range value"<<std::endl;
      steer_amount = 16000;
    }
    
    UIPanel panel;
    MainMenu* menu = new MainMenu(&panel, &actions, &render, audio_device);
    panel.setFrontMenu(menu);
    panel.resizeEvent(w,h);
    panel.setMenu(menu);

    /* Main loop */
    SDL_Event event;
    bool quit = false;
    while (!menu->getQuit() && !quit) {

      // Handle events
      while (!quit && SDL_PollEvent(&event)) {

	if (event.key.keysym.sym == SDLK_ESCAPE || panel.getCurrentMenu())
	  panel.event(event);
	else if (event.type == SDL_QUIT)
	  quit = true;
	else {
	  ActionMapping::const_iterator action = actions.find(event);
	  if (action != actions.end()) {
	    switch(action->second) {
	    case Steering:
	    case Acceleration:
	    case Braking:
	    case AccelerationAndBraking:
            case BrakingAndAcceleration:
		menu->getGameManager()->sendEvent(action->second, Control::getValue(event));
	      break;

	    case SteerLeft:
		menu->getGameManager()->sendEvent(Steering, action->first.invertedEvent(event)?0:-steer_amount);
	      break;
	    case SteerRight:
		menu->getGameManager()->sendEvent(Steering, action->first.invertedEvent(event)?0:steer_amount);
	      break;
	    case Accelerate:
	      	menu->getGameManager()->sendEvent(Acceleration, action->first.invertedEvent(event)?-32767:32767);
	      break;
	    case Brake:
	      	menu->getGameManager()->sendEvent(Braking, action->first.invertedEvent(event)?-32767:32767);
	      break;

	    case NextCamera:
	      if (!action->first.invertedEvent(event))  // We do not want to change the camera on key release
		menu->getGameManager()->nextCamera();
	      break;
	    case PrevCamera:
	      if (!action->first.invertedEvent(event))
		menu->getGameManager()->prevCamera();
	      break;
            case FreeCamera:
              if (!action->first.invertedEvent(event))
                menu->getGameManager()->toggleFreeCamera();
              break;
              
            case FreeCameraHorizontal:
              if (Control::getValue(event) > 8000)
                menu->getGameManager()->freeCamera()->rotateY((Control::getValue(event)-8000)/-8000.0);
              else if (Control::getValue(event) < -8000)
                menu->getGameManager()->freeCamera()->rotateY((Control::getValue(event)+8000)/-8000.0);
              
              menu->getGameManager()->freeCamera()->update();
              break;
              
            case FreeCameraVertical:
              if (Control::getValue(event) > 8000)
                menu->getGameManager()->freeCamera()->rotateX((Control::getValue(event)-8000)/-8000.0);
              else if (Control::getValue(event) < -8000)
                menu->getGameManager()->freeCamera()->rotateX((Control::getValue(event)+8000)/-8000.0);

              menu->getGameManager()->freeCamera()->update();
              break;

            case FreeCameraMoveX:
              if (Control::getValue(event) > 8000) {
                top10::math::Vector T((Control::getValue(event)-8000)/-20000.0, 0.0, 0.0);
                menu->getGameManager()->freeCamera()->translateL(T);
              }
              else if (Control::getValue(event) < -8000) {
                top10::math::Vector T((Control::getValue(event)-8000)/-20000.0, 0.0, 0.0);                
                menu->getGameManager()->freeCamera()->translateL(T);
              }
              
              menu->getGameManager()->freeCamera()->update();
              break;

            case FreeCameraMoveZ:
              if (Control::getValue(event) > 8000) {
                top10::math::Vector T(0.0, 0.0, (Control::getValue(event)-8000)/-20000.0);
                menu->getGameManager()->freeCamera()->translateL(T);
              }
              else if (Control::getValue(event) < -8000) {
                top10::math::Vector T(0.0, 0.0, (Control::getValue(event)-8000)/-20000.0);                
                menu->getGameManager()->freeCamera()->translateL(T);
              }
              
              menu->getGameManager()->freeCamera()->update();
              break;
              
	    default:
	      std::cerr<<"Action "<<action->second<<" not handled yet";
	      break;
	    }
	  }
	}
      	
      }

      if (quit) continue;

      // Update game and display it
      menu->getGameManager()->updateGame();
      render.clearAllGL();
      menu->getGameManager()->renderGL();

      glPushAttrib(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      glDisable(GL_DEPTH_TEST);
      glDisable(GL_ALPHA_TEST);
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();
      glTranslatef(-0.5,0,0);
      glScalef(0.035,0.05,1.0);
      glColor4f(1.0, 1.0, 1.0, 0.8);
      panel.updateGL();
      glPopAttrib();

      SDL_GL_SwapBuffers();
    }
    saveActionMapping(top10::util::UserFile::getPath()+"/inputs", actions);
    top10::util::GlobalOptions::getGlobalOptions()->save(top10::util::UserFile::getPath()+"/options.xml");
  }
  catch (const std::string& e) {
    cerr<<"Fatal error caught: "<<e<<endl;
  }
#if 0
  catch (...) {
    cerr<<"Unknown error caught"<<endl;
  }
#endif
 
 cout<<"Exit"<<endl;
 
 return 0;
}
