/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <fstream>
#include <cmath>

#include "Kart.hh"
#include "helpers/3dsRead.hh"
#include "util/error.hh"
#include "util/PathFinder.hh"
#include "util/Parse.hh"
#include "track/Track.hh"
#include "math/Matrix.hh"
#include "graphX/AutoCameraNode.hh"

using namespace top10::ui_interactive;
using namespace top10::math;
using namespace std;

using top10::helpers::Read3DS;
using top10::util::Error;

/*
 * KartManager
 */

KartManager* KartManager::instance = 0;

KartManager::KartManager() {}

KartManager* KartManager::getInstance()
{
  if (instance) return instance;
  else return (instance = new KartManager);
}

KartNode* KartManager::getKart(std::string model_filename)
{
  // See if we already loaded this kind of kart
  std::map<std::string, top10::util::Ref<KartNode> >::iterator cache_f = cache.find(model_filename);
  if (cache_f != cache.end()) return cache_f->second.getPtr();
  
  // If not, load it now
  KartNode* ret = new KartNode(model_filename);
  cache[model_filename] = ret;
  ret->setName("KART["+model_filename+"]");
  
  return ret;
}

/*
 * KartNode
 */

#define EXPECT(str) if (token != str) throw Error("Expected "+std::string(str)+", got "+token)

KartNode::KartNode(std::string model_filename)
{
  for (int i=0; i<=LAST; ++i) {
    static_transforms[i] = new top10::graphX::TransformNode;
    dynamic_transforms[i] = new top10::graphX::TransformNode;
    dynamic_transforms[i]->addChild(static_transforms[i].getPtr());
    addChild(dynamic_transforms[i].getPtr());
  }
  
  // All 3d models that are part of the kart
  std::map<std::string, top10::util::Ref<top10::graphX::Node> > model_map;
  
  top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
  std::string path = finder.find(model_filename);
  if (path.empty()) throw Error("Could not find ") + model_filename;
  std::ifstream in(path.c_str());
  if (!in) throw Error("Could not open "+path);
  finder.addPath(top10::util::PathFinder::getPath(path));
  
  /* Expected:
     BEGIN MODELS
     <model filename> <alias name>
     ...
     END
  */

  std::string token;
  in>>token;
  EXPECT("BEGIN");
  in>>token;
  EXPECT("MODELS");
  for (;;) {
    token = top10::util::parseString(in);
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    path = finder.find(token);
    if (path.empty()) throw Error("Could not find "+token);
    top10::helpers::Read3DS r3ds(path);
    top10::graphX::Node* model = r3ds.getTopNode();
    if (!model) throw Error("Could not load "+path);

    token = top10::util::parseString(in);
    model_map[token] = model;
  }

  /* Expected:
     BEGIN BODY
     <alias> <object name>
     TRANSFORM <4x4 matrix (16 floats)>
     END
  */
  std::map<std::string, top10::util::Ref<top10::graphX::Node> >::iterator f;

  const char* keywords[] = {"BODY", "WHEEL_FL", "WHEEL_FR", "WHEEL_RL", "WHEEL_RR", "STEERING_WHEEL"};
  std::string aliases[LAST+1];
  std::string objects[LAST+1];
  top10::math::Matrix4 transforms[LAST+1];
  
  for (int i=0; i<=LAST; ++i) {
    in>>token;
    EXPECT("BEGIN");
    parsePart(in, keywords[i], &(aliases[i]), &(objects[i]), &(transforms[i]));
    static_transforms[i]->setToWorld(transforms[i]);
    if (i == KartNode::STEER) {
      Vector steer_center, steer_axis;
      
      in>>token;
      EXPECT("CENTER");
      in>>steer_center.x;
      in>>steer_center.y;
      in>>steer_center.z;

      in>>token;
      EXPECT("AXIS");
      in>>steer_axis.x;
      in>>steer_axis.y;
      in>>steer_axis.z;
      
      double axis_size = steer_axis.size();
      if (axis_size < SMALL_VECTOR) throw Error("Axis of steering wheel is too small");
      steer_axis /= axis_size;
    
      // Compute the transformations that will bring the wheel to lay horizontal around (0,0,0)
      Vector z(0,0,1);
      Vector x = steer_axis^z;
      Matrix4 M(x, steer_axis, z);
      Translation4 T(steer_center);
      M = T*M;
      normal_to_component[KartNode::STEER] = M;
      component_to_normal[KartNode::STEER] = top10::math::inverse(M);
    }
    else if (i == KartNode::BODY) {
      // Nothing, parsePart did everything already.
    }
    else {
      Vector attach;
      
      in>>token;
      EXPECT("CENTER");
      in>>attach.x>>attach.y>>attach.z;
    
      normal_to_component[i] = Translation4(attach);
      component_to_normal[i] = Translation4(-attach);
    }

    in>>token;
    EXPECT("END");

    f = model_map.find(aliases[i]);
    if (f == model_map.end()) throw Error("Bad alias "+aliases[i]);
    top10::graphX::Node* object_node = f->second->find(objects[i]);
    if (object_node == 0) throw Error("Bad object name "+objects[i]);
    static_transforms[i]->addChild(object_node);
  }
}

KartNode* KartNode::clone() const
{
  KartNode* ret = new KartNode(*this);
  for (int i=0; i<=LAST; ++i) {
    ret->removeChild(dynamic_transforms[i].getPtr());
    ret->dynamic_transforms[i] = new top10::graphX::TransformNode;
    ret->dynamic_transforms[i]->addChild(ret->static_transforms[i].getPtr());
    ret->addChild(ret->dynamic_transforms[i].getPtr());
  }
  return ret;
}

void KartNode::parsePart(std::istream& in, const char* keyword, std::string* alias, std::string* object, top10::math::Matrix4* M)
{
  using top10::util::parseString;

  std::string token;

  in>>token;
  EXPECT(keyword);
  in>>token;
  EXPECT("MODEL");
  *alias = parseString(in);
  *object = parseString(in);
  in>>token;
  EXPECT("TRANSFORM");
  in>>(*M);
}

void KartNode::update(double steer_angle,
                      double steer_fl, double steer_fr,
                      double h_fl, double h_fr,
                      double h_rl, double h_rr) const
{
  top10::math::Matrix4 R = top10::math::Identity4();
  double c = cos(steer_angle);
  double s = sin(steer_angle);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  dynamic_transforms[STEER]->setToWorld(normal_to_component[STEER]*R*component_to_normal[STEER]);
  
  c = cos(steer_fl);
  s = sin(steer_fl);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  R(1,3) = h_fl;
  dynamic_transforms[WHEEL_FL]->setToWorld(normal_to_component[WHEEL_FL]*R*component_to_normal[WHEEL_FL]);

  c = cos(steer_fr);
  s = sin(steer_fr);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  R(1,3) = h_fr;
  dynamic_transforms[WHEEL_FR]->setToWorld(normal_to_component[WHEEL_FR]*R*component_to_normal[WHEEL_FR]);
  
  R(0,0) =  R(2,2) = 1;
  R(2,0) = R(0,2) = 0;
  R(1,3) = h_rl;
  dynamic_transforms[WHEEL_RL]->setToWorld(normal_to_component[WHEEL_RL]*R*component_to_normal[WHEEL_RL]);
  R(1,3) = h_rr;
  dynamic_transforms[WHEEL_RR]->setToWorld(normal_to_component[WHEEL_RR]*R*component_to_normal[WHEEL_RR]);
}

/*
 * Kart
 */
const double Kart::memory_factor = 10.0;

Kart::Kart(top10::track::Track* track, Vector _center, Vector e1):
  top10::physX::Kart(track, _center, e1),
  friction_left_accum(0.0),
  friction_right_accum(0.0),
  friction_left_diff(0.0),
  friction_right_diff(0.0),
  load_left_accum(0.0),
  load_right_accum(0.0),
  side_force_accum(0.0),
  friction_force_fl(0,0,0),
  friction_force_fr(0,0,0),
  friction_force_rl(0,0,0),
  friction_force_rr(0,0,0)
{
}

void Kart::updateUI(double dt)
{
  if (dt == 0.0) return;
  
  double old_val = friction_left_accum;
  friction_left_accum = (memory_factor * friction_left_accum
			 + wheel_rl.getFriction() /*+ wheel_fl.getFriction()*/)
    / (1+memory_factor);
  friction_left_diff *= memory_factor;
  friction_left_diff += (friction_left_accum - old_val)/dt;
  friction_left_diff /= (1+memory_factor);

  old_val = friction_right_accum;
  friction_right_accum = (memory_factor * friction_right_accum
			  + wheel_rr.getFriction() /*+ wheel_fr.getFriction()*/)
    / (1+memory_factor);
  friction_right_diff *= memory_factor;
  friction_right_diff += (friction_right_accum - old_val)/dt;
  friction_right_diff /= (1+memory_factor);

  load_left_accum *= memory_factor;
  load_left_accum += wheel_rl.getLoad() /*+ wheel_fl.getLoad()*/;
  load_left_accum /= (1+memory_factor);

  load_right_accum *= memory_factor;
  load_right_accum += wheel_rr.getLoad() /*+ wheel_fr.getLoad()*/;
  load_right_accum /= (1+memory_factor);

  double mom_mem = 0.9;
  double mom_new = 0.1;
  //  side_force_accum = (mom_mem * side_force_accum + mom_new * body.getForce() * right) / (mom_new + mom_mem);

  friction_force_fl = (friction_force_fl * memory_factor + wheel_fl.getFrictionForce());
  friction_force_fr = (friction_force_fr * memory_factor + wheel_fr.getFrictionForce());
  friction_force_rl = (friction_force_rl * memory_factor + wheel_rl.getFrictionForce());
  friction_force_rr = (friction_force_rr * memory_factor + wheel_rr.getFrictionForce());

  
}

double Kart::getLeftFriction() const
{
  if (friction_left_diff>0.0)
    return friction_left_diff;
  else return 0.0;
}

double Kart::getRightFriction() const
{
  if (friction_right_diff>0.0)
    return friction_right_diff;
  else return 0.0;
}

double Kart::getLeftLoad() const
{
  return load_left_accum;
}

double Kart::getRightLoad() const
{
  return load_right_accum;
}

int Kart::getNCameras() const
{
  return 6;
}

top10::graphX::CameraNode* Kart::getCamera(int i) const
{
  const top10::math::Matrix3& orient(body.getOrientation());
  Vector right = orient * Vector(0,0,1);
  Vector forward = orient * Vector(1,0,0);
  Vector up = orient * Vector(0,1,0);
  Vector center = body.getMassCenter();
  double fov = 30;

  Vector pos, dir, dup;
  switch(i) {
  case 0: //In
    pos = center - forward*0.5 + Vector(0, 0.45,0);
    dir = forward - up*0.2;
    dup = Vector(0,1,0);
    fov = 45;
    break;
  case 1: //Front
    pos = center + forward*5 + Vector(0,1,0);
    dir = -forward - Vector(0,0.2,0);;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 2: //Back
    pos = center - forward*5 + Vector(0,1.5,0);
    dir = forward - Vector(0,0.2,0);
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 3: //left
    pos = center - right*15 + Vector(0,1,0);
    dir = right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 4: //right
    pos = center + right*15 + Vector(0,1,0);
    dir = -right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 5: //top
    pos = center + Vector(0,50,0);
    dir = Vector(0,-1,0);
    fov = 50;
    dup = forward;
    break;
  default: return 0;
  };

  top10::graphX::AutoCameraNode* camera = new top10::graphX::AutoCameraNode;
  camera->accessView().setCenter(pos);
  camera->accessView().setDirection(dir, dup);
  camera->accessView().setFOV(fov);
  camera->accessView().setNear(0.7);
  camera->accessView().setFar(500);
  camera->accessView().setRatio(1.4); //FIXME
  camera->accessView().update();
  
  return camera;
}


top10::racing::KartState Kart::getKartState() const
{
  top10::racing::KartState ret;
  ret.timestamp = 0;
  ret.orient = body.getOrientation();
  ret.translation = body.getTranslation();
  ret.steer = steer_angle;

  return ret;
}

