/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "GameManager.hh"
#include "util/SlowTime.hh"
#include "util/GlobalOptions.hh"
#include "graphX/LightNode.hh"
#include "graphX/AlphaNode.hh"
#include "graphX/CullNode.hh"
#include "graphX/Wireframe.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/ShadowVolumeNode.hh"

#include <plib/fnt.h>

namespace top10 {
namespace ui_interactive {

GameManager::GameManager(top10::graphX::Renderer* render, AudioDevice* audio): render(render), audio(audio), started(false), use_freecam(false)
{
  free_camera = new top10::graphX::FreeCameraNode;

  std::string font_name = getOptS("Render.Font");    
  std::string font_path = top10::util::PathFinder::defaultPathFinder().find("fonts/"+font_name+".txf");
  if (font_path.empty()) throw std::string("Could not load font fonts/"+font_name+".txf");
  font = new fntTexFont(font_path.c_str());
}

void GameManager::createTrainingSession(std::string track_name, std::string kart_model_name, bool wet_weather)
{
  if (audio) {
    SDL_PauseAudio(1);
    SDL_LockAudio();
  }
  
  terminateGame();
  camera_kart = camera = 0;
  
  track = top10::track::loadTrack(track_name);
  
  kart_model = KartManager::getInstance()->getKart(kart_model_name);
  simulation = new Simulation(track.getPtr());
  if (audio) audio->setKart(simulation->getKart(0));
    
  //Setup the rendering
  top10::graphX::CullNode* cull = new top10::graphX::CullNode;
  cull->setCulling(top10::graphX::RenderState::Back);
  cull->addChild(track->getTopNode());
  
  skybox = new top10::graphX::SkyBoxNode;
  track_and_karts = new top10::graphX::LightNode;
  track_and_karts->addChild(cull);
  
  if (getEnabledOpt("Render.Shadow.Method")) {
    shadows = new top10::graphX::GroupNode;
  }
  
  Kart* kart;
  int i=0;
  while ( (kart = simulation->getKart(i++)) ) {
    // Create the transformation for the position of the kart
    top10::util::Ref<top10::graphX::TransformNode> trans(new top10::graphX::TransformNode);
    trans->setToWorld(getKartTransform(kart));
    
    // The node of the kart
    top10::util::Ref<KartNode> kart_node(kart_model->clone());
    trans->addChild(kart_node.getPtr());
    kart_nodes.push_back(kart_node.getPtr());
    
    // Keep track of the transformation (and the kart)
    karts.push_back(trans.getPtr());
    
    // Cull the kart's back faces
    cull->addChild(trans.getPtr());
    
    // Optionaly, create a shadow
    if (getOptS("Render.Shadow.Method") == "Volume")
      shadows->addChild(makeProxy(trans.getPtr(), new top10::graphX::ShadowVolumeNode::MyProxyOperation, true));
  }

  // The ghost
  ghost_alpha = new top10::graphX::AlphaNode;
  ghost_alpha->setAlpha(getOptUC("Render.Ghost.Alpha"));
  ghost_trans = new top10::graphX::TransformNode;
  ghost = kart_model->clone();
  ghost_trans->addChild(ghost.getPtr());
  ghost_alpha->addChild(ghost_trans.getPtr());
  
  // The hud needs to be slightly transparent, otherwise transparent bits in the font texture won't show at all
  top10::graphX::AlphaNode* hud_alpha = new top10::graphX::AlphaNode;
  hud_alpha->setAlpha(getOptUC("Render.HUD.Alpha"));
  hud = hud_alpha;
  
  // Use screen pixel coordinates
  top10::graphX::TransformNode* hud_transform = new top10::graphX::TransformNode;
  top10::math::Matrix4 translate = top10::math::Translation4(top10::math::Vector(-1.0, -1.0, 0.0));
  top10::math::Matrix4 scale = top10::math::Scaling3(1.0/render->getScreenWidth(), 1.0/render->getScreenHeight(), 1.0);
  translate *= scale;
  hud_transform->setToWorld(translate);
  hud->addChild(hud_transform);
  
  // The timers, showing the current and best times
  current_time = new top10::graphX::TextNode;
  current_time->setFont(font);
  current_time->setPos(getOptD("Render.HUD.CurrentTime.Pos.X"), getOptD("Render.HUD.CurrentTime.Pos.Y"));
  current_time->setPointSize(getOptD("Render.HUD.CurrentTime.Font.Size"));
  hud_transform->addChild(current_time.getPtr());
  
  best_time = new top10::graphX::TextNode;
  best_time->setFont(font);
  best_time->setPos(getOptD("Render.HUD.BestTime.Pos.X"), getOptD("Render.HUD.BestTime.Pos.Y"));
  best_time->setPointSize(getOptD("Render.HUD.BestTime.Font.Size"));
  hud_transform->addChild(best_time.getPtr());
  
  last_time = new top10::graphX::TextNode;
  last_time->setFont(font);
  last_time->setPos(getOptD("Render.HUD.LastTime.Pos.X"), getOptD("Render.HUD.LastTime.Pos.Y"));
  last_time->setPointSize(getOptD("Render.HUD.LastTime.Font.Size"));
  hud_transform->addChild(last_time.getPtr());
      
  if (audio) {
    SDL_UnlockAudio();
    SDL_PauseAudio(0);  
  }
}

void GameManager::updateGame()
{
  if (!started) return;
  
  unsigned int now = top10::util::global_time.getTicks();
  simulation->update(now - time_start);
  
  Kart* kart;
  int i=0;
  while ( (kart = simulation->getKart(i)) ) {
    top10::util::Ref<top10::graphX::TransformNode> trans(karts[i]);
    trans->setToWorld(getKartTransform(kart));
    kart_nodes[i]->update(kart->getSteer(), kart->getLeftSteer(), kart->getRightSteer(),
                          kart->getHeightFL(), kart->getHeightFR(), kart->getHeightRL(), kart->getHeightRR());
    kart->updateUI((now - last_update)/1000.0);
    ++i;
  }
  
  const top10::racing::KartTimer* timer = simulation->getTimer(camera_kart);
  if (timer) {
    if (ghost_hidden) {
      ghost_hidden = false;
      track_and_karts->addChild(ghost_alpha.getPtr());
    }
    
    current_time->setTime(timer->getCurrentTime());
    best_time->setTime(timer->getBestTime());
    last_time->setTime(timer->getLastTime());
    
    // Update the ghost
    if (simulation->getBestLap().size() != 0) {
      top10::racing::KartState s = simulation->getBestLap().getKartState(timer->getCurrentTime());
      ghost_trans->setToWorld(getKartTransform(s));
      ghost->update(s.steer, s.steer, s.steer, 0.0, 0.0, 0.0, 0.0);
    }
  }
  
  last_update = now;
}

top10::math::Matrix4 GameManager::getKartTransform(const Kart* kart) const
{
  top10::racing::KartState s = kart->getKartState();
  top10::math::Matrix4 T = top10::math::Translation4(-kart->getMassCenterL());
  top10::math::Matrix4 T_inv = top10::math::Translation4(kart->getMassCenterL());
  top10::math::Matrix4 O = s.orient;
  top10::math::Matrix4 P = top10::math::Translation4(s.translation);
  return P*T_inv*O*T;
}

top10::math::Matrix4 GameManager::getKartTransform(top10::racing::KartState s) const
{
  top10::math::Matrix4 O = s.orient;
  top10::math::Matrix4 P = top10::math::Translation4(s.translation);
  return P*O;
}

void GameManager::startGame()
{
  if (simulation.isValid()) {
    started = true;
    last_update = time_start = top10::util::global_time.getTicks();
  }
}

void GameManager::terminateGame()
{
  started = false;
  ghost_hidden = true;
  if (simulation.isValid()) {
    /* Not necessary to discard those explicitely, since they will be required the next time a session is created.
      It just helps free some memory before some more is allocated to load the new models, textures...
    */
    track.discard();
    kart_model.discard();
    skybox.discard();
    simulation.discard();
    track_and_karts.discard();
    hud.discard();
    shadows.discard();
    
    // We must clear those arrays
    kart_nodes.clear();
    karts.clear();
  }  
}

void GameManager::sendEvent(Action a, Sint16 value)
{
  if (simulation.isValid())
    simulation->sendEvent(a, value);    
}

void GameManager::renderGL()
{
  if (!simulation.isValid()) return;
  
  render->clearAllGL();
  render->setCamera(getCamera());

  if (getEnabledOpt("Render.Skybox")) {
    render->clearList();
    render->buildList(skybox.getPtr());
    render->renderSkyBoxGL();
    render->clearDepthGL();
  }

  // Draw karts outlines
  if (getEnabledOpt("Render.Kart.Outline")) {
    for (int i = 0; i<karts.size(); ++i) {
      top10::util::Ref<top10::graphX::Node> outline;
      if (getOptS("Render.Kart.Outline") == "PolyFill") {
        top10::graphX::WireframeFill* wf = new top10::graphX::WireframeFill;
        wf->setThickness(getOptD("Render.Kart.Outline.Thickness"));
        wf->addChild(karts[i]);
        outline = wf;
      }
      else if (getOptS("Render.Kart.Outline") == "Wireframe") {
        top10::graphX::Wireframe* wf = new top10::graphX::Wireframe;
        wf->addChild(karts[i]);
        wf->setThickness(getOptD("Render.Kart.Outline.Thickness"));
        outline = wf;
      }
      else throw std::string("Unsupported outline method ")+getOptS("Render.Kart.Outline");
      
      top10::graphX::CullNode* cull = new top10::graphX::CullNode;
      cull->setCulling(top10::graphX::RenderState::Front);
      cull->addChild(outline.getPtr());
    
      top10::graphX::MaterialNode* color = new top10::graphX::MaterialNode;
      color->r = getOptUC("Render.Kart.Outline.Color.Red");
      color->g = getOptUC("Render.Kart.Outline.Color.Green");
      color->b = getOptUC("Render.Kart.Outline.Color.Blue");
      color->addChild(cull);

      render->clearList();    
      render->buildList(color);
      render->renderGL();
      delete color;
    }
  }

  // Draw the karts and the track
  render->clearList();
  render->buildList(track_and_karts.getPtr());
  render->renderGL();

  // If shadows are enabled
  if (shadows.isValid() && shadows->getChild(0)) {
    // Update the shadow volumes
    top10::graphX::ShadowVolumeNode::recurseUpdate(top10::math::Vector(0, 100, 0), shadows.getPtr());
    // Create a culling node
    top10::util::Ref<top10::graphX::CullNode> cull(new top10::graphX::CullNode);
    // Cull back faces of the shadow volumes
    cull->setCulling(top10::graphX::RenderState::Back);
    cull->addChild(shadows.getPtr());
  
    // enable stencil test, disable rendering
    render->toggleColorRendering(false);
    render->toggleDepthRendering(false);
    render->enableStencil(GL_ALWAYS, 0);
    
    // Render front faces into the stencil buffer
    render->clearList();
    render->buildList(cull.getPtr());
    render->setStencilOp(GL_KEEP, GL_KEEP, GL_INCR);
    render->renderGL();
    
    // Render back faces into the stencil buffer
    cull->setCulling(top10::graphX::RenderState::Front);
    render->clearList();
    render->buildList(cull.getPtr());
    render->setStencilOp(GL_KEEP, GL_KEEP, GL_DECR);  
    render->renderGL();
    
    // Render the "shadows": a transparent 2d black panel
    top10::util::Ref<top10::graphX::PanelNode> panel(new top10::graphX::PanelNode);
    top10::util::Ref<top10::graphX::AlphaNode> alpha(new top10::graphX::AlphaNode);
    alpha->setAlpha(getOptUC("Render.Shadow.Alpha"));
    alpha->addChild(panel.getPtr());
    panel->setDist(0);
  
    render->toggleColorRendering(true);
    render->enableStencil(GL_NOTEQUAL, 0);
    render->clearList();
    render->buildList(alpha.getPtr());
    render->toggleDepthTest(false);
    render->renderHudGL();
    
    // Restore the rendering flags
    render->toggleDepthRendering(true);
    render->toggleDepthTest(true);
    render->disableStencil();
  }

  if (getEnabledOpt("Render.HUD")) {
    render->clearDepthGL();
    render->clearList();
    render->buildList(hud.getPtr());
    render->renderHudGL();
  }

}

void GameManager::nextCamera()
{
  if (!simulation.isValid()) return;
  
  ++camera;
  Kart* kart = simulation->getKart(camera_kart);
  if (!kart) return;
  
  if (camera == kart->getNCameras()) camera = 0;
}

void GameManager::prevCamera()
{
  if (!simulation.isValid()) return;
  
  --camera;
  if (camera < 0) camera = 0;
}

void GameManager::nextKart()
{
  if (!simulation.isValid()) return;
  
  ++camera_kart;
  if (camera_kart == simulation->getNKarts()) camera_kart = 0;   
}

void GameManager::prevKart()
{
  if (!simulation.isValid()) return;
  
  --camera_kart;
  if (camera_kart < 0) camera_kart = 0;    
}

top10::graphX::CameraNode* GameManager::getCamera()
{
  if (!simulation.isValid()) return 0;
  
  if (use_freecam) return free_camera.getPtr();
  
  return simulation->getKart(camera_kart)->getCamera(camera);
}

}
}
