/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_FORCE_FEEDBACK_HH
#define TOP10_FORCE_FEEDBACK_HH

#include <string>

#ifdef HAVE_FF
#include <linux/input.h>
#endif

#include "Kart.hh"
#include "util/error.hh"
#include "math/Curve.hh"

namespace top10 {
  namespace ui_interactive {

    /**
       Class handling interaction with a force feedback device.
       This class is an abstract class. Different kinds of devices (wheels, joysticks, mice...)
       implement this interface
    */
    class FFDevice {
    public:
      enum effect {
      	None=0,
	Bumps=1,
	Centrifugal=2,
	Steer_drag=4,
	Steer_center=8,
	All=15
      };

    protected:
      int enabled_effects;

    public:
      /**
	 Create a FFdevice, but do not initialise it.
	 You have to call setKart() with a valid pointer to a Kart before use.
      */
      FFDevice(): enabled_effects(None), kart(0) {}
	
      /**
	 Create a FFdevice and assign it a kart. Can immediately be used, then
      */
      FFDevice(Kart* _kart): enabled_effects(All), kart(_kart) {}

      /**
	 Set the kart. Force effects are computed and rendered with respect to
	 a kart.
      */
      inline void setKart(Kart* _kart) { kart = _kart; }

      /**
	 Enable/Disable an effect
      */
      inline bool toggleEffect(effect fx) { tellEffectToggled(fx); enabled_effects = enabled_effects ^ fx; return (enabled_effects & fx) != 0; }
      virtual void tellEffectToggled(effect fx) =0;

      /**
	 Update the continuous effects (engine vibration...) and generate
	 shock effects
      */
      virtual void render(int ms_elapsed) =0;

      virtual ~FFDevice() {};

    protected:
      Kart* kart;
    };

    class FFJoystick: public FFDevice {
    public:
      FFJoystick(int device_number) throw(top10::util::Error);

      void render(int);
      void tellEffectToggled(FFDevice::effect);

      virtual ~FFJoystick();

    private:
      int fd;
      /// Magnitude of momentum on steering wheel = f(momentum on kart)
      top10::math::Curve2D centrifugal_magnitude;

#ifdef HAVE_FF
      ff_effect bumps;
      bool bumps_playing;
      int centrifugal_wait_time;
      static const int centrifugal_update_time = 500;
      ff_effect centrifugal;
      ff_effect steer_friction;
      ff_effect center_back;
      static const int n_effects = 4;
#endif

    private:
      FFJoystick(const FFJoystick&) {/*Forbidden*/}
    };

    /// For now, let's say wheels and joysticks can be handled the same way
    typedef FFJoystick FFWheel;

    /// Does not do anything for now.
#if 0
    class FFMouse: public FFdevice {
    };
#endif

    FFDevice* ff_factory(const std::string& device_name, int);

  };	// ui_interactive
};	// top10

#endif
