/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#ifdef WIN32
#include <windows.h>
#endif

#include <SDL/SDL.h>
#include <GL/glu.h>
#include <plib/pu.h>

#include "util/PathFinder.hh"
#include "util/UserFile.hh"
#include "graphX/Renderer.hh"
#include "graphX/LightNode.hh"
#include "graphX/GridNode.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/SkyBoxNode.hh"
#include "graphX/AimMarkNode.hh"
#include "../Drawable.hh"
#include "Camera.hh"
#include "DialogManager.hh"
#include "MeshDialog.hh"
#include "FileMenuDialog.hh"
#include "StartingAreaDialog.hh"
#include "DecorationsDialog.hh"
#include "VisualDialog.hh"
#include "CheckpointsDialog.hh"
#include "SurfacesDialog.hh"

using namespace std;
using top10::math::Vector;

/*
 * Error dialog box
 */
static puDialogBox* error_dialog = 0;
static void error_cb(puObject* ob)
{
  puDeleteObject(error_dialog);
  error_dialog = 0;
}

static void make_dialog(std::string message)
{
  std::cerr<<message<<std::endl;

  if (error_dialog) return;

  error_dialog = new puDialogBox ( 50, 50 ) ;
  {
    puFrame* frame = new puFrame ( 0, 0, 400, 100 ) ;
    puText* text = new puText( 10, 70);
    text->setLabel(message.c_str());

    puOneShot *ok = new puOneShot ( 180, 10, "OK" ) ;
    ok -> makeReturnDefault ( TRUE ) ;
    ok -> setCallback       ( error_cb ) ;
  }
  error_dialog->close  () ;
  error_dialog->reveal () ;
}

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );      
  screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
  if (screen == NULL) {
    throw string("Could not set GL mode: ") + string(SDL_GetError());
  }

  return screen;
}

int main(int argc, char** argv)
{
  using namespace top10::util;
  using namespace top10::graphX;
  using namespace top10::tracked;
  
  try {
    int screen_width = 800;
    int screen_height = 600;
    string filename;

    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (string(argv[i]) == "-w") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_width = atoi(argv[i]);
      }
      else if (string(argv[i]) == "-h") {
	++i;
	if (i >= argc) throw string("Missing argument");
	screen_height = atoi(argv[i]);
      }
      else {
	filename = argv[i];
      }
    }

    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw string("Couldn't initialize SDL: ") + string(SDL_GetError());

    SDL_Surface* screen = initGL(0 /* Depth not specified */, screen_width, screen_height, false /* No fullscreen */);
    SDL_EnableUNICODE(1);

    // PUI init
    puInit();
    puSetDefaultStyle(PUSTYLE_SMALL_SHADED);

    // Build the camera
    Ref<Camera> camera(new Camera);
    camera->setScreenSize(screen_width, screen_height);

    // Setup all the track editors
    MainEditor main_ed;
    MeshEditor mesh_ed;
    SurfacesEditor surf_ed;
    DecorationsEditor deco_ed;
    CheckpointsEditor cps_ed;
    StartingAreaEditor sa_ed;

    main_ed.setMeshEditor(&mesh_ed);
    main_ed.setSurfacesEditor(&surf_ed);
    main_ed.setDecorationsEditor(&deco_ed);
    main_ed.setCheckpointsEditor(&cps_ed);
    main_ed.setStartingAreaEditor(&sa_ed);
    surf_ed.setMeshEditor(&mesh_ed);
    deco_ed.setMeshEditor(&mesh_ed);
    cps_ed.setMeshEditor(&mesh_ed);
    sa_ed.setMeshEditor(&mesh_ed);
    mesh_ed.setCamera(camera.getPtr());

    // The scene graph of the track
    Ref<top10::graphX::MaterialNode> top_node(new MaterialNode);
    top_node->r = top_node->g = top_node->b = 255;
    
    top_node->addChild(mesh_ed.getNode());
    top_node->addChild(sa_ed.getNode());
    top_node->addChild(cps_ed.getNode());
    top_node->addChild(surf_ed.getNode());    
    top_node->addChild(deco_ed.getNode());  
      
    Ref<GridNode> grid(new GridNode);
    Ref<MaterialNode> glowing(new MaterialNode);
    glowing->g = glowing->glow_b = 255;
    glowing->addChild(grid.getPtr());
    top_node->addChild(glowing.getPtr());
    
    Ref<LightNode> light_node(new LightNode);
    light_node->addChild(top_node.getPtr());

    // SG of the skybox
    Ref<SkyBoxNode> skybox(new SkyBoxNode);
    
    // SG of the HUD
    Ref<GroupNode> hud_node(new GroupNode);
    Ref<AimMarkNode> mark_node(new AimMarkNode);
    hud_node->addChild(mark_node.getPtr());
    
    // The dialog manager and the dialogs
    DialogManager dialog_manager(&main_ed, &mesh_ed, &surf_ed, &deco_ed, &cps_ed, &sa_ed);
    FileMenuDialog file_dialog(&dialog_manager);
    MeshDialog mesh_dialog(&dialog_manager);
    StartingAreaDialog sa_dialog(&dialog_manager);
    DecorationsDialog deco_dialog(&dialog_manager);
    VisualDialog visual_dialog(&dialog_manager);
    CheckpointsDialog cps_dialog(&dialog_manager);
    SurfacesDialog surf_dialog(&dialog_manager);

    // The renderer
    Renderer renderer;
    renderer.setCamera(camera.getPtr());
    renderer.initGL(screen_width, screen_height);
        
    // Main loop
    SDL_Event event;
    for (bool leave = false; !leave; ) {
      renderer.clearAllGL();

      renderer.clearList();
      renderer.buildList(skybox.getPtr());
      renderer.renderSkyBoxGL();
      
      renderer.clearDepthGL();
      renderer.clearList();
      renderer.buildList(light_node.getPtr());
      renderer.renderGL();
      
      renderer.clearDepthGL();
      renderer.clearList();
      renderer.buildList(hud_node.getPtr());
      renderer.renderHudGL();
      
      // Draw the PUI widgets
      glPushAttrib(GL_CURRENT_BIT);
      puDisplay();
      glPopAttrib();
      
      SDL_GL_SwapBuffers();

      // Handle events
      SDL_WaitEvent(0);
      while (!leave && SDL_PollEvent(&event)) {
	switch (event.type) {
	case SDL_KEYDOWN:
	  if (event.key.keysym.sym == SDLK_ESCAPE) {
	    dialog_manager.toggleActive();
	  }
	  else if (dialog_manager.isActive()) {
	    try {
	      // Forward event to pui
	      switch (event.key.keysym.sym) {
	      case SDLK_RIGHT: puKeyboard(PU_KEY_RIGHT, PU_DOWN); break;
	      case SDLK_LEFT:  puKeyboard(PU_KEY_LEFT, PU_DOWN); break;
	      default: 	     puKeyboard(event.key.keysym.unicode & 0x7f, PU_DOWN); break;
	      }
	    }
	    catch(std::string e) {
	      make_dialog(e);
	    }
	  }
          else {
            camera->handleEvent(event);   
          }
	  break;
	    
	case SDL_QUIT:
	  leave = true;
	  break;
	    
	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
	  if (dialog_manager.isActive()) {
	    // Forward to pui
	    int button=0;
	    if      (event.button.button == 1) button = PU_LEFT_BUTTON;
	    else if (event.button.button == 2) button = PU_MIDDLE_BUTTON;
	    else if (event.button.button == 3) button = PU_RIGHT_BUTTON;

	    int state=0;
	    if      (event.button.state == SDL_PRESSED)  state = PU_DOWN;
	    else if (event.button.state == SDL_RELEASED) state = PU_UP;

	    try {
	      puMouse(button, state, event.button.x, event.button.y);
	    }
	    catch(std::string e) {
	      make_dialog(e);
	    }

	  }
	  else camera->handleEvent(event);
	  break;

	case SDL_MOUSEMOTION:
	  if (!dialog_manager.isActive()) {
	    camera->handleEvent(event);
	  }
	  break;

	default:
	  break;
	}
      }
    }
  }
  catch (string& err) {
    cerr<<"Exception: "<<err<<endl;
    SDL_Quit();
    exit(1);
  }
  catch (...) {
    cerr<<"Unknown exception"<<endl;
    SDL_Quit();
    exit(1);
  }

  SDL_Quit();
  exit(0);
}
