/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "MeshEditor.hh"
#include "helpers/OrientMatrix.hh"
#include "helpers/3dsRead.hh"
#include "graphX/IntersectionFinder.hh"

using namespace top10::tracked;

MeshEditor::MeshEditor():
  scaling(1.0), axis(top10::math::Y), axis_negate(false), mesh_node(0), transform_node(0), camera(0)
{
  transform_node = new top10::graphX::TransformNode;
  getNode()->addChild(transform_node.getPtr());
}

MeshEditor::~MeshEditor()
{
  getNode()->removeChild(transform_node.getPtr());
}

void MeshEditor::loadMesh(std::string filename)
{
  top10::helpers::Read3DS r3ds(filename);  
  model_filename=filename;
  
  transform_node->removeChild(mesh_node.getPtr());
  mesh_node = r3ds.getTopNode();
  transform_node->addChild(mesh_node.getPtr());
  
  updateTransform();
}

void MeshEditor::updateTransform()
{
  top10::math::Matrix4 M = top10::helpers::makeOrientation(axis, axis_negate, scaling);
  transform_node->setToWorld(M);
}

void MeshEditor::reset()
{
  //Do nothing
}

std::string MeshEditor::getPickedMesh() const
{
  assert(camera.isValid());
  if (!mesh_node.isValid()) return std::string();

  // Compute intersection with the mesh
  top10::math::Ray ray;
  ray.setInfinite();
  ray.setOrigin(camera->getView().getCenter());
  ray.setDirection(camera->getView().getDirection());
  top10::graphX::IntersectionFinder finder(transform_node.getPtr(), ray);
  if (finder.getResult()->empty()) return "";
  else return finder.getResult()->begin()->second->getName();
}

std::string MeshEditor::getPickedSurface() const
{
  using top10::math::Vector;
  assert(camera.isValid());
  if (!mesh_node.isValid()) return std::string();
  
  // Compute intersection with the mesh
  top10::math::Ray ray;
  ray.setInfinite();
  ray.setOrigin(camera->getView().getCenter());
  ray.setDirection(camera->getView().getDirection());
  top10::graphX::IntersectionFinder finder(transform_node.getPtr(), ray);
  if (finder.getResult()->empty()) return "";
  else return finder.getResult()->begin()->second->getMaterialName();
}

top10::math::Vector MeshEditor::getPickedPoint() const
{
  using top10::math::Vector;
  Vector pt;
  Vector dir = camera->getView().getDirection();
  Vector pos = camera->getView().getCenter();
  if (fabs(dir.y) < 0.01) throw std::string("Aiming too far");
  double t = -pos.y/dir.y;
  if (t < 0) throw std::string("Aiming away from the ground");
  pt = pos+t*dir;
  return pt;
}
