/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "CheckpointsEditor.hh"
#include <cassert>

using namespace top10::tracked;

CheckpointsEditor::CheckpointsEditor():
  top_node(0),
  mesh_ed(0)
{
  top_node = new Node(this);
  getNode()->addChild(top_node.getPtr());
  reset();
}

CheckpointsEditor::~CheckpointsEditor()
{
  getNode()->removeChild(top_node.getPtr());
}

void CheckpointsEditor::reset()
{
  current_cp = -1;
}

void CheckpointsEditor::pickLeft()
{
  assert(mesh_ed);
  assert(isCorrect(current_cp));

  if (current_cp == -1) throw std::string("You must create a new checkpoint first");
  
  top10::math::Vector p = mesh_ed->getPickedPoint();
  checkpoints.at(current_cp).setPoint1(p);
}

void CheckpointsEditor::pickRight()
{
  assert(mesh_ed);
  assert(isCorrect(current_cp));

  if (current_cp == -1) throw std::string("You must create a new checkpoint first");
  
  top10::math::Vector p = mesh_ed->getPickedPoint();
  checkpoints.at(current_cp).setPoint2(p);
}

void CheckpointsEditor::addNew()
{
  assert(isCorrect(current_cp));
  checkpoints.insert(checkpoints.begin()+current_cp+1, top10::racing::CheckPoint());
}

void CheckpointsEditor::remove()
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) throw std::string("Cannot remove before first checkpoint");
  checkpoints.erase(checkpoints.begin()+current_cp);
  --current_cp;
  assert(isCorrect(current_cp));
}

void CheckpointsEditor::next()
{
  assert(isCorrect(current_cp));
  if (current_cp+1 == (int)checkpoints.size()) throw std::string("Already on last checkpoint");
  else ++current_cp;
  assert(isCorrect(current_cp));
}

void CheckpointsEditor::previous()
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) throw std::string("Already before first checkpoint");
  else --current_cp;
  assert(isCorrect(current_cp));
}

top10::racing::CheckPoint CheckpointsEditor::getCurrent() const
{
  assert(isCorrect(current_cp));
  if (current_cp == -1) return top10::racing::CheckPoint();
  else return checkpoints.at(current_cp);
}

void CheckpointsEditor::Node::renderGL(const top10::graphX::RenderingFeatures&,
  const top10::graphX::RenderState&,
  const top10::graphX::CameraNode&) const
{
  using top10::racing::CheckPoints;
  using top10::racing::CheckPoint;
  using top10::math::Vector;

  assert(ed->isCorrect(ed->current_cp));

  // Mark the current checkpoint
  if (ed->current_cp >= 0) {
    CheckPoint cp = ed->checkpoints.at(ed->current_cp);
    Vector p1 = cp.getPoint1();
    Vector p2 = cp.getPoint2();

    glBegin(GL_LINE_STRIP);
    glVertex3f(p1.x, p1.y, p1.z);
    glVertex3f(p1.x, p1.y + 5, p1.z);
    glVertex3f(p2.x, p2.y + 5, p2.z);
    glVertex3f(p2.x, p2.y, p2.z);
    glEnd();
  }

  // Draw other checkpoints
  for (CheckPoints::const_iterator cp = ed->checkpoints.begin();
       cp != ed->checkpoints.end();
       ++cp) {
    Vector p1 = cp->getPoint1();
    Vector p2 = cp->getPoint2();
      
    glBegin(GL_LINE_STRIP);
    glVertex3f(p1.x, p1.y, p1.z);
    glVertex3f(p1.x, p1.y + 5, p1.z);
    glVertex3f(p2.x, p2.y + 5, p2.z);
    glVertex3f(p2.x, p2.y, p2.z);
    glEnd();
  }
}

//! Check the checkpoint index is in range
bool CheckpointsEditor::isCorrect(int cp) const
{
  return (cp >= -1) && (cp+1 <= (int)checkpoints.size());
}
