/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_TRACK_TRACK_HH
#define TOP10_TRACK_TRACK_HH

#include "math/Vertex.hh"
#include "math/AxisEnum.hh"
#include "math/Mesh.hh"
#include "math/Triangle.hh"
#include "helpers/GenericOctree.hh"
#include "util/Debug.hh"
#include "util/PathFinder.hh"
#include "util/RefCount.hh"
#include "racing/StartingArea.hh"
#include "racing/CheckPoint.hh"
#include "graphX/GroupNode.hh"
#include "graphX/MeshNode.hh"
#include "physX/SurfacePatch.hh"
#include "physX/World.hh"
#include <set>
#include <string>
#include <cassert>

namespace top10 {

  namespace track {

    /**
       Interface of all tracks.
    */
    class Track: public top10::physX::FixedObject, public top10::util::Debug, public top10::util::RefCount
    {
    public:
      Track(top10::graphX::Node* meshes, const std::set<std::string>& deco_mesh_names,
            const std::set<std::string>& road_surfaces,
            const std::set<std::string>& dusty_surfaces,
            const std::set<std::string>& border_surfaces,
            const std::set<std::string>& dirt_surfaces,
            const std::set<std::string>& grass_surfaces,
            const std::set<std::string>& sand_surfaces);

      ~Track();
      
      void setStartingArea(top10::racing::StartingArea);
      void setCheckPoints(top10::racing::CheckPoints);

      inline top10::racing::StartingArea* getStartingArea() { return &starting_area; }
      inline top10::racing::CheckPoints* getCheckPoints() { return &checkpoints; }
      inline const top10::racing::StartingArea* getStartingArea() const { return &starting_area; }
      inline const top10::racing::CheckPoints* getCheckPoints() const { return &checkpoints; }

      /*!
	Compute the plane containing the surface patch situated under p
	\param p
	\param patch output parameter
	\return false if no surface patch could be found
      */
      bool getSurfacePatch(const top10::math::Vector& p, top10::physX::SurfacePatch& patch) const;

      std::vector<top10::math::Triangle> getPotentialCollisions(const top10::math::Box&) const;
      
      /**
	 Build the structure that allows such a fast triangle retrievial ;)
      */
      void buildTriMap(int depth=5);

      //! Set the global friction factor
      /*! getSurfacePatch mulitplies the "stickyness" by this grip factor.
	Can be used to simulate wet or icy surfaces */
      void setGripFactor(double f) { friction_factor = f; }
      double getGripFactor() const { return friction_factor; }

      inline top10::graphX::Node* getTopNode() const { return top_node.getPtr(); }
      
    private:
      top10::util::Ref<top10::graphX::Node> top_node;
      top10::racing::StartingArea starting_area;
      top10::racing::CheckPoints checkpoints;

      struct FaceRef {
        FaceRef(const top10::math::Mesh* m, int f, int s): mesh(m), face_idx(f), surf_idx(s) {}
        const top10::math::Mesh* mesh;
        int face_idx;
        int surf_idx;      
      };
      
      typedef std::vector<FaceRef> FaceRefSet;
      typedef std::vector<FaceRefSet> SetRow;
      typedef std::vector<SetRow> TriMap;
      
      static inline top10::math::Vector toVector(FaceRef face, int idx) {
        assert(idx>=0 && idx<3);
        return face.mesh->getVertices()->at( (face.mesh->getFaces()->at(face.face_idx)).idxs[idx] );
      }
      
      static inline void toVectors(FaceRef face, top10::math::Vector* out) {
        top10::math::Mesh::Face f = face.mesh->getFaces()->at(face.face_idx);
        for (int i=0; i<3; ++i) out[i] = face.mesh->getVertices()->at(f.idxs[i]);
      }
      
      const FaceRefSet* debugTriMap(const top10::math::Vector&) const;

      double friction_factor;
      std::vector<const top10::math::Mesh*> meshes;
      
      static const int ROAD_IDX=0, DUST_IDX=1, BORDER_IDX=2, DIRT_IDX=3, GRASS_IDX=4, SAND_IDX=5;
      top10::physX::Surface surfaces[6];
      
      /**
	 This is a 2d array. Every cell contains a list of triangles.
	 It allows to quick-test which triangle intersects with a given vertical
	 line. */
      TriMap trimap;

      void iterateBuild();
      static bool cellIntersect(FaceRef, double, double, double);
      static bool inPolygon(FaceRef, double, double);
      static bool inPolygon(top10::math::Vector*, double, double);
      static void subdivide(const FaceRefSet& triangles, double x, double z, double size,
			    FaceRefSet& ll, FaceRefSet& lr,
			    FaceRefSet& ul, FaceRefSet& ur);
      static bool diag1_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s);
      static bool diag2_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s);

      double cell_size;

      /// Coordinates of the bounding box
      top10::math::Vector min_v;
      top10::math::Vector max_v;
      
      //! Octree of all decorative triangles
      typedef top10::helpers::GenericOctree<top10::math::Triangle> Octree;
      Octree deco_octree;
    };

    //! Load a new track, oriented and scaled correctly
    Track* loadTrack(std::string filename);

    //! Load the parts of a track (mesh, orientation...) without any correction
    void loadTrackParts(std::string filename,
			std::string* meshname, top10::math::AxisEnum *axis, bool* axis_neg, double* scale,
			top10::racing::StartingArea* area, top10::racing::CheckPoints* cps,
			std::set<std::string>* decorative_meshes,
			std::set<std::string>* road_surfaces,
			std::set<std::string>* dusty_surfaces,
			std::set<std::string>* border_surfaces,
			std::set<std::string>* dirt_surfaces,
			std::set<std::string>* grass_surfaces,
			std::set<std::string>* sand_surfaces);

    //! Counter-part of loadTrackParts.
    void saveMeshTrack(std::string filename,
		       std::string meshname, top10::math::AxisEnum axis, bool axis_neg, double scale,
		       top10::racing::StartingArea area, top10::racing::CheckPoints cps,
		       std::set<std::string> decorative_meshes,
		       std::set<std::string> road_surfaces,
		       std::set<std::string> dusty_surfaces,
		       std::set<std::string> border_surfaces,
		       std::set<std::string> dirt_surfaces,
		       std::set<std::string> grass_surfaces,
		       std::set<std::string> sand_surfaces);
  };
};

#endif
