/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_KART_HH
#define TOP10_KART_HH

#include "ComplexObject.hh"
#include "RigidBody.hh"
#include "Wheel.hh"
#include "Engine.hh"
#include "World.hh"
#include "Point.hh"

namespace top10 {

  namespace track {
    class Track;
  }

  namespace physX {

    class Kart: public ComplexObject
    {
      public:
      static double STEER_ANGLE;
      static double STEER_RADIUS;
      static double TYRE_MASS;
      static double TYRE_WIDTH;
      static double DRIVER_MASS;
      static double KART_MASS;
      static double FRONT_X;
      static double FRONT_Y;
      static double FRONT_Z;
      static double BACK_X;
      static double BACK_Y;
      static double BACK_Z;
      static double FRONT_INERTIA;
      static double BACK_INERTIA;
      static double ENGINE_INERTIA;
      static double FRONT_RADIUS;
      static double BACK_RADIUS;
      static double FRONT_LONG_K;
      static double BACK_LONG_K;
      static double FRONT_SIDE_K;
      static double BACK_SIDE_K;
      static double FRONT_STIFFNESS;
      static double BACK_STIFFNESS;
      static double FRONT_DAMPING;
      static double BACK_DAMPING;
      static double FRONT_MAX_DAMPING_FORCE;
      static double BACK_MAX_DAMPING_FORCE;
      static double ENGINE_RPS_MAX;
      static double ENGINE_RPS_OPT;
      static double ENGINE_FT_TORQUE0;
      static double ENGINE_FT_TORQUE1;
      static double ENGINE_NT_TORQUE1;
      static double ENGINE_NT_TORQUE2;
      static double BRAKE_K;
      static double COLLISION_FACTOR;

    public:
      /**
         Reads characteristics of a Kart from a file.
         Those characteristics include:
         - Size of kart
         - Characteristics of engine
         - Brakes
         - Tyres
	 - ...
      */
      static void initDefaultParameters(std::istream&);

      /**
	 Creates a new kart.
	 \param track Pointer to an existing track
	 \param starting_pos Where to place the kart
	 \param orientation The initial orientation of the kart
      */
      Kart(top10::track::Track* track,
	   Vector starting_pos=Vector(0, 0, 0),
	   Vector orientation=Vector(-1, 0, 0));

      void collideTrack(double dt);
      void handleCollisions(const std::vector<top10::math::Triangle>& polyset, double dt);
      void update(double G, double dt);
      void integrate(double dt);

      /**
	 Get registered into a world
	 \param world World into which the kart is to be registered
      */
      void getRegistered(World& world);

      //! Return the bounding box in global coords
      top10::math::Box getBoundingBox() const;

      /**
	 Sets the position of the gas pedal.
	 \parameter a Value between 0 and 1 representing the gas pedal position.
      */
      void accel(double a);

      /**
	 Sets the position of the brakes pedal.
	 \parameter b Value between 0 and 1.
      */
      void brake(double b);

      /**
	 Sets the angle of the steering wheel.
	 \parameter fraction Value between -1 (right) and 1 (left)
      */
      void steer(double fraction);

      /**
	 Gets the position of the center of the kart.
      */
      inline Vector getPos() const {return body.getMassCenter();}

      inline Vector getMassCenterL() const {return body.getMassCenterL();}
      /**
         Gets the speed of the kart.
      */
      inline Vector getSpeed() const {return body.getSpeedAtL(body.getMassCenterL());}
      
      //TODO: check which members should be moved from ui_interactive to here,
      //if any
      /**
	 Gets the RPM of the engine, but in radians per second
      */
      double getRadPerSec() const;

      /**
	 Get the maximum RPM of the engine, in rad per second
      */
      double getMaxRadPerSec() const;

      /**
         Get the average bumpness of the surface under the wheels
      */
      double getBumps() const;
      
      virtual ~Kart() {};

    protected:
      //! Anchor points for the wheels
      Vector fl, fr, rl, rr, top;
      Wheel wheel_fl, wheel_fr, wheel_rl, wheel_rr;
      Engine engine;
      RigidBody body;
      top10::math::Box original_bbox;
      //! Boxes used for collision response
      top10::math::AxisAlignedBox fl_box, fr_box, rl_box, rr_box, front_box, back_box, left_box, right_box;

      //! Vertical speeds of the wheels (in the kart's coordinate system)
      //      double vspeed_fl, vspeed_fr, vspeed_rl, vspeed_rr;

      bool is_colliding;

      double steer_angle;

    private:
      Vector computeLinearImpulse(Vector speed, Vector normal) const;
      Vector computeImpulse(Vector speed, Vector normal, Vector pos) const;
      void updateActive();
    };
  };
};

#endif
