/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef ENGINE_H
#define ENGINE_H

#include "math/Curve.hh"
#include "util/Dumpable.hh"
#include "Wheel.hh"
#include <cassert>

namespace top10 {
  namespace physX {
    /**
       Describes an engine and brakes.
       Supposed to be connected to two wheels
    */
    class Engine: top10::util::Dumpable
    {
    public:
      Engine(Wheel* left, Wheel* right,
	     double rps_max,
	     double rps_opt,
	     double ft_torque0,
	     double ft_torque1,
	     double nt_torque1,
	     double nt_torque2,
	     double brake_k);

      void update(double dt);

      inline void brake(double val) { assert(val>=0); assert(val<=1.0); braking = val;}
      inline double getBrake() const { return braking; }
      
      inline void accelerate(double val) { assert(val>=0); assert(val<=1.0); accel = val;}
      inline double getAccel() const { return accel; }

      double getAngularVelocity() const { return w_speed; }
      void setAngularVelocity(double w) { w_speed = w; }

      void dumpTo(std::ostream&) const;
      void loadFrom(std::istream&);

      inline double getTorque() const {return torque;}

      //! Check if the wheels are locked.
      /*! Used for the ABS driving help */
      bool wheelsLocked(double dt) const;
      
    private:
      //! Forbidden
      Engine& operator=(const Engine& other) { assert(0); return *(Engine*)0; }

      Wheel* const left_wheel;
      Wheel* const right_wheel;
      /// *** Values for accel = 100%
      Curve2D full_accel_torque;
      /// *** Values for accel = 0%
      Curve2D no_accel_torque;
      double braking_torque;
      bool enable_ABS;
      //! Minimal value of angular acceleration, under which we consider the wheels are locked
      double locked_threshold;

      double accel;
      double braking;
      
      //! Angular velocity
      double w_speed;
      double old_w_speed;
      
      double torque;
    };

  };
};
#endif
