/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <SDL.h>
#include <GL/glu.h>

#include "util/PathFinder.hh"
#include "helpers/3dsRead.hh"
#include "FreeCameraNode.hh"
#include "GroupNode.hh"
#include "LightNode.hh"
#include "Renderer.hh"
#include "GridNode.hh"
#include "SkyBoxNode.hh"
#include "ShadowVolumeNode.hh"

using namespace top10::graphX;

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
#if 1
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
#else
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 0 );
#endif
  screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
  if (screen == NULL) {
    throw std::string("Could not set GL mode: ") + std::string(SDL_GetError());
  }

  return screen;
}

int main(int argc, char** argv)
{
  try {
    int width = 800;
    int height = 600;
    std::string filename;
    
    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (std::string(argv[i]) == "-w") {
	++i;
	if (i >= argc) throw std::string("Missing value: width");
	width = atoi(argv[i]);
      }
      else if (std::string(argv[i]) == "-h") {
	++i;
	if (i >= argc) throw std::string("Missing value: height");
	height = atoi(argv[i]);
      }
      else {
	filename = argv[i];
      }
    }
    if (filename.empty()) throw std::string("Missing filename");      
    
    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw std::string("Couldn't initialize SDL: ") + std::string(SDL_GetError());
    SDL_Surface* screen = initGL(0 /* Depth not specified */, width, height, false /* No fullscreen */);

    // GL init
    Renderer render;
    render.initGL(width, height);
    
    // Find the location of the model file
    top10::util::PathFinder path_finder = top10::util::PathFinder::defaultPathFinder();
    std::string path = path_finder.find(filename);
    if (path.empty()) {
      throw std::string("Could not find ")+filename;
    }
    
    // Load the file
    top10::helpers::Read3DS model3ds(path.c_str());
    Node* model = model3ds.getTopNode();
 
    // Create a shadow volume
    Node* shadow_volumes = makeProxy(model, new ShadowVolumeNode::MyProxyOperation, true);
    
    // Create a camera    
    FreeCameraNode* camera = new FreeCameraNode;
    FreeCameraNode* camera1 = new FreeCameraNode;
    FreeCameraNode* camera2 = new FreeCameraNode;
    FreeCameraNode* camera3 = new FreeCameraNode;
    FreeCameraNode* camera4 = new FreeCameraNode;
    render.setCamera(camera);
    
    // Create a light
    LightNode* light = new LightNode;
    ShadowVolumeNode::recurseUpdate(light->getLight().getPosition(), shadow_volumes);
    
    // Create a glowing grid representing the ground
    GridNode* grid = new GridNode;
    MaterialNode* glowing = new MaterialNode;
    glowing->b = 255;
    glowing->glow_b = 255;
    glowing->addChild(grid);
    
    // Put everyone into one graph
    GroupNode* top = new GroupNode;
    top->addChild(light);
    top->addChild(camera);
    top->addChild(glowing);
    glowing->addChild(shadow_volumes);
    light->addChild(model);

    // Create a skybox
    SkyBoxNode* skybox = new SkyBoxNode;
        
    SDL_Event event;
    for (bool leave = false; !leave;) {
      // Draw the scene
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      
      render.clearList();
      render.buildList(skybox);
      render.renderSkyBoxGL();
      glClear(GL_DEPTH_BUFFER_BIT);
      
      render.clearList();
      render.buildList(top);
      render.renderGL();

      SDL_GL_SwapBuffers();
      
      // Handle events
      while (SDL_PollEvent(&event)) {
	switch (event.type) {
      	case SDL_KEYDOWN:
	  if (event.key.keysym.sym == SDLK_ESCAPE) leave=true;
          else if (event.key.keysym.sym == SDLK_SPACE) {
            Light new_light = light->getLight();
            new_light.setPosition(camera->getView().getCenter());
            new_light.setDirection(camera->getView().getDirection());
            light->setLight(new_light);
            ShadowVolumeNode::recurseUpdate(camera->getView().getCenter(), shadow_volumes);
          }
          else if (event.key.keysym.sym == SDLK_F1) {
            *camera1 = *camera;
          }
          else if (event.key.keysym.sym == SDLK_F2) {
            *camera2 = *camera;
          }
          else if (event.key.keysym.sym == SDLK_F3) {
            *camera3 = *camera;
          }
          else if (event.key.keysym.sym == SDLK_F4) {
            *camera4 = *camera;
          }
          
          else if (event.key.keysym.sym == SDLK_F5) {
            *camera = *camera1;
          }
          else if (event.key.keysym.sym == SDLK_F6) {
            *camera = *camera2;
          }
          else if (event.key.keysym.sym == SDLK_F7) {
            *camera = *camera3;
          }
          else if (event.key.keysym.sym == SDLK_F8) {
            *camera = *camera4;
          }
          
	  break;

      	case SDL_QUIT:
      	  leave = true;
          break;

      	case SDL_MOUSEMOTION:
	  if (event.motion.state == 1) {
            camera->rotateY(90.0 * event.motion.xrel / height);
            camera->rotateX(90.0 * event.motion.yrel / width);
	  }
	  if (event.motion.state > 1) {
	    camera->translateL(top10::math::Vector(-50.0*event.motion.xrel/width, 0.0, 50.0*event.motion.yrel/height));
	  }
          camera->update();

	  break;
      		
      	default:
      	  break;
      	}
      }
    }
  }
  catch (std::string err) {
    std::cerr<<err<<std::endl;
    return 1;
  }
  
  return 0;
}
