/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "ShadowVolumeNode.hh"

namespace top10 {
namespace graphX {

ShadowVolumeNode::ShadowVolumeNode(const top10::math::Mesh* m): outline(m), vertex_array(0), normal_array(0)
{
}

ShadowVolumeNode::~ShadowVolumeNode()
{
  delete[] vertex_array;
  delete[] normal_array;
}

void ShadowVolumeNode::update(top10::math::Vector light_pos)
{
  using top10::math::Vector;
  
  edges = outline.compute(light_pos);
  
  delete[] vertex_array;
  vertex_array = new float[1+edges.size()*12];
  
  delete[] normal_array;
  normal_array = new float[1+edges.size()*12];

  const std::vector<top10::math::Vector>* vertices = outline.getMesh()->getVertices();
    
  int c=0;
  for (std::list<top10::math::Outline::Edge>::const_iterator edge_it = edges.begin(); edge_it != edges.end(); ++edge_it) {
    Vector v[4];
    if (edge_it->inverted) {
      v[0] = vertices->at(edge_it->p2);
      v[1] = vertices->at(edge_it->p1);
    }
    else {
      v[0] = vertices->at(edge_it->p1);
      v[1] = vertices->at(edge_it->p2);
    }
    v[2] = v[1] + 10.0 * (v[1] - light_pos);
    v[3] = v[0] + 10.0 * (v[0] - light_pos);
    Vector n = (v[1] - v[0]) ^ (v[1] - light_pos);  // No need to normalize, we only care about the orientation

    for (int j=0; j<4; ++j)
      for (int i=0; i<3; ++i) {
        vertex_array[c] = v[j][i];
        normal_array[c] = n[i];
        ++c;
      }
  }
  
}

void ShadowVolumeNode::renderGL(const RenderingFeatures& features, const RenderState& s, const CameraNode& camera) const
{
  assert(vertex_array);
  assert(normal_array);
  using top10::math::Vector;

  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glVertexPointer(3, GL_FLOAT, 0, (const GLvoid*)vertex_array);
  glNormalPointer(GL_FLOAT, 0, (const GLvoid*)normal_array);
  glDrawArrays(GL_QUADS, 0, 2*edges.size());
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);  
}

void ShadowVolumeNode::recurseUpdate(top10::math::Vector light_pos, Node* n)
{
  top10::math::Matrix4 I = top10::math::Identity4();
  recurseUpdate(light_pos, n, I);   
}
      
void ShadowVolumeNode::recurseUpdate(top10::math::Vector light_pos, Node* n, top10::math::Matrix4 M)
{
  ShadowVolumeNode* vn = dynamic_cast<ShadowVolumeNode*>(n);
  if (vn) {
    M = inverse(M);
    vn->update(M*light_pos);
  }
  else {
    TransformNode* tn = dynamic_cast<TransformNode*>(n);
    if (tn) M = M*tn->toWorld(); 
  
    else {
      TransformNodeProxy* tn2 = dynamic_cast<TransformNodeProxy*>(n);
      if (tn2) M = M*tn2->toWorld();
    }
  }
  
  Node* child;
  int i=0;
  while ((child=n->getChild(i++))) recurseUpdate(light_pos, child, M);
}
  
Node* ShadowVolumeNode::MyProxyOperation::makeChild(const MeshNode* m) const
{
  return new ShadowVolumeNode(m->getMesh());
}

}
}
