/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifdef WIN32
#include <windows.h>
#endif

#include "RenderState.hh"
#include "TextureManager.hh"
#include <GL/gl.h>
#include <GL/glext.h>

#include <cassert>

using namespace top10::graphX;

RenderState::RenderState():
  fixed_flags(0),
  alpha(255), r(0), g(0), b(0), glow_r(0), glow_g(0), glow_b(0), texture_id(0), filtering(None), offset(0),
  front_is_direct(true), culling(NoCulling), line_width(1.0), poly_fill_mode(Full)
{
  to_world = top10::math::Identity4();
}

void RenderState::addLight(Light l)
{
  if ((fixed_flags & LightsBit) == 0)
    lights.push_back(l);  
}

bool RenderState::operator<(const RenderState& other) const
{
  // 0 Sort according to transparency (draw opaque objects first)
  if (alpha != other.alpha) return other.alpha < alpha;
  
  // 1 Sort wrt textures
  if (texture_id != other.texture_id) return texture_id < other.texture_id;

  // 2 Sort wrt material props
#define TMP_SORT(r)\
  if (r != other.r) return r < other.r
  TMP_SORT(r);
  TMP_SORT(g);
  TMP_SORT(b);
  TMP_SORT(glow_r);
  TMP_SORT(glow_g);
  TMP_SORT(glow_b);
#undef TMP_SORT

  // 3. lighting environment.
  Lights::const_iterator it1, it2;
  it1 = lights.begin();
  it2 = other.lights.begin();
  while (it1 != lights.end() && it2 != other.lights.end()) {
    if (*it1 != *it2) return *it1 < *it2;
    ++it1;
    ++it2;
  }
  if (it1 != lights.end()) return false;
  if (it2 != other.lights.end()) return true;
  
  // 4 transforms.
  for (int j=0; j<4; ++j) {
    // We assume the last line is always (0 0 0 1)
    for (int i=0; i<3; ++i) {
      if (to_world(i,j) != other.to_world(i,j)) return to_world(i,j) < other.to_world(i,j);
    }
  }

  // 5 line width
  if (other.line_width != line_width) return other.line_width < line_width;
  
  // 6 fill mode
  if (other.poly_fill_mode != poly_fill_mode) return other.poly_fill_mode < poly_fill_mode;
  
  // 7 depth offset  
  if (other.offset != offset) return other.offset < offset;
  
  // 8 culling
  if (other.front_is_direct != front_is_direct) return front_is_direct;
  if (other.culling != culling) return other.culling < culling;

  return false;
}

void RenderState::setStateGL(const RenderState& old_state, const RenderingFeatures& features, bool force_init) const
{
  // Texture mapping
  if (force_init || texture_id != old_state.texture_id) {
    if (texture_id != 0) { // Enable texturing
      glEnable(GL_TEXTURE_2D);
    } else /*if (texture_id == 0)*/ { // Disable texturing
      glDisable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, 0);
    }
    
    if (texture_id != 0 && texture_id != TextureManager::other_textures_id) { // Set the texture object
      assert(glIsTexture(texture_id));
      glBindTexture(GL_TEXTURE_2D, texture_id);
    }
  }
  if (force_init || (texture_id != 0 && ((filtering != old_state.filtering) || (texture_id != old_state.texture_id)))) {
    switch(filtering) {
    case None:
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      break;
    case Linear:
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      break;
    case Bilinear:
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
      break;
    case Trilinear:
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
      break;
    case Anisotropic:
      if (!features.hasAnisotropicFiltering) {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      }
#ifdef GL_TEXTURE_MAX_ANISOTROPY_EXT
      else {      
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, features.maxAnisotropy);
      }
#else
#warning "GL_TEXTURE_MAX_ANISOTROPY_EXT undefined"
#endif
      break;
    }
  }
  
  // Depth-buffer read-only when drawing transparent objects
  if (force_init || old_state.alpha != alpha) {
    if (alpha != 255) {
      glAlphaFunc(GL_LESS, 1.0);
      glDepthMask(GL_FALSE);        
    }
    else {
      glAlphaFunc(GL_EQUAL, 1.0);
      glDepthMask(GL_TRUE);
    }
  }

  // Material
  if (force_init || r != old_state.r || g != old_state.g || b != old_state.b || alpha != old_state.alpha) {
    float v[] = {r/255.0, g/255.0, b/255.0, alpha/255.0};
#ifndef NDEBUG
    for (int i=0; i<4; ++i) {
      assert(v[i] >= 0.0);
      assert(v[i] <= 1.0);
    }
#endif     
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, v);
    glColor4f(v[0], v[1], v[2], v[3]);
  }

  if (force_init || glow_r != old_state.glow_r || glow_g != old_state.glow_g || glow_b != old_state.glow_b || alpha != old_state.alpha) {
    float v[] = {glow_r/255.0, glow_g/255.0, glow_b/255.0, alpha/255.0};
#ifndef NDEBUG
    for (int i=0; i<4; ++i) {
      assert(v[i] >= 0.0);
      assert(v[i] <= 1.0);
    }
#endif         
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, v);
  }
  
  // Transformation matrix
  if (force_init || to_world != old_state.to_world) {
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
    glPushMatrix();
    double d[16];
    to_world.toGL(d);
    glMultMatrixd(d);
  }
  
  // Lighting
  if ((force_init || !old_state.lights.empty()) && lights.empty()) glDisable(GL_LIGHTING);
  else if ((force_init || old_state.lights.empty()) && !lights.empty()) glEnable(GL_LIGHTING);
  
  Lights::const_iterator it1 = lights.begin();
  Lights::const_iterator it2 = old_state.lights.begin();
  int i=0;
  while (it1 != lights.end() && it2 != old_state.lights.end()) {
    if (force_init || *it1 != *it2)
      it1->setLightGL(GL_LIGHT0 +i);
    ++it1;
    ++it2;
    ++i;
  }

  while (it1 != lights.end()) {
    it1->setLightGL(GL_LIGHT0 +i);
    ++it1;
    ++i;
  }
  while (it2 != old_state.lights.end()) {
    glDisable(GL_LIGHT0 +i);
    ++it2;
    ++i;  
  }
  
  // Line width
  if (force_init || old_state.line_width != line_width) {
    glLineWidth(line_width);
  }
  
  // Fill mode
  if (force_init || old_state.poly_fill_mode != poly_fill_mode) {
    glPolygonMode(GL_FRONT_AND_BACK, poly_fill_mode==Full? GL_FILL:GL_LINE);    
  }
  
  // Depth Offset
  if (force_init || old_state.offset != offset) {
    if (offset != 0.0) {
      glEnable(GL_POLYGON_OFFSET_FILL);
      glPolygonOffset(0, offset);
    }
    else {
      glDisable(GL_POLYGON_OFFSET_FILL);
    }
  }
  
  // Culling
  if (force_init || old_state.front_is_direct != front_is_direct) {
    if (front_is_direct) glFrontFace(GL_CCW);
    else glFrontFace(GL_CW);
  }
  
  if (force_init || old_state.culling != culling) {
    if (culling == NoCulling)
      glDisable(GL_CULL_FACE);
    else {
      glEnable(GL_CULL_FACE);
      if (culling == Front) glCullFace(GL_FRONT);
      else glCullFace(GL_BACK);
    }
  }
  
#ifndef NDEBUG
  checkConsistency();
#endif
}

void RenderState::checkConsistency() const
{
  float v[4];
  glGetMaterialfv(GL_FRONT, GL_AMBIENT, v);
  float _r = v[0];
  float _g = v[1];
  float _b = v[2];
  float _alpha = v[3];
  
  glGetMaterialfv(GL_FRONT, GL_EMISSION, v);
  float _glow_r = v[0];
  float _glow_g = v[1];
  float _glow_b = v[2];
  float _alpha2 = v[3];
  
  GLint _texture_id;
  if (glIsEnabled(GL_TEXTURE_2D))
    glGetIntegerv(GL_TEXTURE_BINDING_2D, &_texture_id);
  else _texture_id = 0;

  bool passed = true;  

#define CHECK_CONSIST(name, value, expected)\
if ((value) !=  (expected)) {\
  std::cerr<<"Consistency check for "<<name<<" failed. Expected "<<expected<<", got "<<value<<std::endl;\
  passed = false;\
}

#define ROUND(f) (((int)((f)*100.0))/100.0)

  CHECK_CONSIST("red", ROUND(_r), ROUND(r/255.0));
  CHECK_CONSIST("green", ROUND(_g), ROUND(g/255.0));
  CHECK_CONSIST("blue", ROUND(_b), ROUND(b/255.0));
  CHECK_CONSIST("glow red", ROUND(_glow_r), ROUND(glow_r/255.0));
  CHECK_CONSIST("glow green", ROUND(_glow_g), ROUND(glow_g/255.0));
  CHECK_CONSIST("glow blue", ROUND(_glow_b), ROUND(glow_b/255.0));
  CHECK_CONSIST("alpha", ROUND(_alpha), ROUND(alpha/255.0));
  
  if (texture_id != TextureManager::other_textures_id) {
    CHECK_CONSIST("texture id", _texture_id, texture_id);
  }
#undef ROUND
#undef CHECK_CONSIST
  if (!passed) {
    abort();
  }
}
