/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "NodeIterator.hh"
#include "TransformNode.hh"

using namespace top10::graphX;

DFSNodeIterator::DFSNodeIterator(Node* n)
{
  pos.push_back(std::make_pair(n,0));
}

DFSNodeIterator& DFSNodeIterator::operator++()
{
  if (atEnd()) throw std::string("Cannot iterate past last element");

  std::pair<Node*, int> top = pos.back();
  for(;;) {
    Node* child = top.first->getChild(top.second);
    if (child) {
      pos.push_back(std::make_pair(child, 0));
      return *this;
    }
    else {
      pos.pop_back();
      if (pos.empty()) return *this;
      top = pos.back();
      pos.pop_back();
      top.second++;
      pos.push_back(top);
    }
  }
}

Node* DFSNodeIterator::operator*() const
{
  if (atEnd()) throw std::string("Cannot dereference past last element");
  return pos.back().first;
}

bool DFSNodeIterator::atEnd() const
{
  return pos.empty();
}

/* ****************************************************************************
   LeafNodeIterator
   ****************************************************************************/

LeafNodeIterator::LeafNodeIterator(Node* node): DFSNodeIterator(node)
{
  // Move to the first leaf
  while (!atEnd() && !onLeaf())
    DFSNodeIterator::operator++();
}

LeafNodeIterator& LeafNodeIterator::operator++()
{
  DFSNodeIterator::operator++();
  while (!atEnd() && !onLeaf())
    DFSNodeIterator::operator++();
  return *this;    
}

Node* LeafNodeIterator::operator*() const
{
  return DFSNodeIterator::operator*();
}

bool LeafNodeIterator::atEnd() const
{
  return DFSNodeIterator::atEnd();
}

top10::math::Matrix4 LeafNodeIterator::getTransform() const
{
  //TODO: improve time by storing the transform matrix along with the nodes in the pos vector
  top10::math::Matrix4 ret = top10::math::Identity4();
  for (std::vector< std::pair<Node*, int> >:: const_iterator path_it = pos.begin(); path_it != pos.end(); ++path_it) {
    const TransformNode* t = dynamic_cast<TransformNode*>(path_it->first);
    if (t) ret = ret*t->toWorld();
  }
  return ret;
}

bool LeafNodeIterator::onLeaf() const
{
  return this->DFSNodeIterator::operator*()->getChild(0) == 0;
}
