/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_DEBUG_H
#define TOP10_DEBUG_H

#include <string>
#include <iostream>
#include <set>

namespace top10 {
  namespace util {
    
    /*!
      A simple class indicating if an object should live in debug mode
    */
    class Debug {
    public:
      /*!
	If debug is true, the constructor prints a debug message to cerr, to
	notify the creation of an object
      */
      Debug(const char* objectName="Unnamed", bool debug=
#ifndef NDEBUG
	    true
#else
	    false
#endif
	    );

      /*!
	Return true if debugging is activated for this object
      */
      inline bool is_debug() const { return _debug; }

      //! Enable/disable debugging
      bool toggle() const { return _debug = !_debug; }

      /*!
	Toggle debugging for this object.
	This function is virtual so that the derived class can enable
	debugging for their subclasses too (for example).
	By default, it toggles debugging for this object.
	
	\return the state of debugging after toggling
      */
      virtual bool toggleDebug() const { return toggle(); }

      /*!
	Return the name of the object
      */
      inline std::string getName() const { return _name; }

      void printDebugStart() const;
      void printDebugEnd() const;

      /*!
	If _debug is true, print a message on cerr notifying the destruction
	of this object
      */
      virtual ~Debug();

    public:
      static std::ostream& debug_stream;
      static std::set<const Debug*> members;

    private:
      mutable bool _debug;
      std::string _name;
    };

  };
};

#endif
