/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

//C++ stuff
#include <iostream>
#include <fstream>

//SDL Stuff
#include <SDL/SDL.h>
#include <SDL/SDL_image.h>

//My stuff
#include "initPaths.hh"
#include "Simulation.hh"
#include "Options.hh"
#include "audio.hh"
#include "ForceFeedback.hh"
#include "UIPanel.hh"
#include "MainMenu.hh"
#include "Controls.hh"
#include "math/Curve.hh"
#include "helpers/LwRead.hh"
#include "util/CleanPtr.hh"
#include "util/PathFinder.hh"
#include "util/UserFile.hh"
#include "util/SlowTime.hh"

extern "C" {
#include "extras/sdl2gl.h"
}
static GLfloat black_color[] =  {0.0f, 0.0f, 0.0f, 1.0f};
static GLfloat fog_color[] = {0.6f, 0.6f, 0.6f, 1.0f};
static GLfloat blue_sky[] =  {0.4f, 0.4f, 1.0f, 1.0f};
static GLfloat gray_sky[] =  {0.6f, 0.6f, 0.6f, 1.0f};

using namespace top10::ui_interactive;
using namespace std;

top10::track::Track* parseNewTrack(std::string);

static void close_joy(int, void* joy)
{
  SDL_JoystickClose((SDL_Joystick*)joy);
}

SDL_Surface* splash()
{
  using namespace top10::ui_interactive::options;

  SDL_Surface* screen;

  screen = SDL_SetVideoMode(256, 256, 0, SDL_NOFRAME);
  if (screen) {
    SDL_Surface* image;

    /* Load the BMP file into a surface */
    std::string image_path = top10::util::PathFinder::find("splash.bmp");
    if (!image_path.empty()) return 0;

    image = SDL_LoadBMP(image_path.c_str());
    if (!image) return image;

    /* Blit onto the screen surface */
    if(SDL_BlitSurface(image, NULL, screen, NULL) < 0)
      fprintf(stderr, "BlitSurface error: %s\n", SDL_GetError());

    SDL_UpdateRect(screen, 0, 0, image->w, image->h);

    /* Free the allocated BMP surface */
    SDL_FreeSurface(image);
  }

  return screen;
}

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  using top10::util::Error;

  SDL_Surface* screen;
  int bits_per_channel(5);

  /* Video init */
  if (depth == 0) {
    // Try 24 bpp
    if (SDL_VideoModeOK( w, h, 24, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0))) {
      depth = 24;
      bits_per_channel = 8;
    }
    // Try 16 bpp
    else if (SDL_VideoModeOK( w, h, 16, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0))) {
      depth = 16;
      bits_per_channel = 5;
    }
    else depth = -1;
  }

  if (depth != -1) {
    cout<<"depth = "<<depth<<" bpc = "<<bits_per_channel<<endl;
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, depth );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );      
    screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
    if (screen == NULL) {
      throw Error(string("Could not set GL mode: ") + string(SDL_GetError()));
    }
  }
  else {
    throw Error("No true color video mode available.");
  }

  // Set viewport
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(30.0, ((GLdouble)w)/h, 0.5, 1000.0);

  // Other GL settings
  glMatrixMode(GL_MODELVIEW);
  glEnable(GL_DEPTH_TEST);
  // TODO: light setting does not belong here
  GLfloat light_position[4] = {1.0, 1.0, 1.0, 0.0};
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  // Blending
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  // Fog
  glFogfv(GL_FOG_COLOR, fog_color);
  glFogi(GL_FOG_MODE, GL_LINEAR);
  glFogf(GL_FOG_START, 5.0);
  glFogf(GL_FOG_END, 100.0);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);  

  return screen;
}

/// Deinitialize SDL when destroyed
class SDL_Destroyer
{
public:
  static const int quit =1;
  static const int sound =2;
  static const int joystick =4;

public:
  SDL_Destroyer(): do_quit(false), do_joy(false), do_sound(false) {};

  /// Tell what parts of the sdl to deinit
  void enable(int what)
  {
    if (what & quit) do_quit = true;
    if (what & sound) do_sound = true;
    if (what & joystick) do_joy = true;
  }

  ~SDL_Destroyer()
  {
    // BUG: SDL deinitializers crash !
    if (do_sound) {
      SDL_CloseAudio();
    }
    //    if (do_joy) cerr<<"TODO: close joy"<<endl;
    if (do_quit) SDL_Quit();
  }

private:
  bool do_quit;
  bool do_joy;
  bool do_sound;
};


int main(int argc, char** argv)
{
  using namespace top10::ui_interactive::options;
  using top10::util::Error;
  using top10::util::CleanPtr;

  ifstream* rep_in(0);
  ofstream* rec_out(0);

  try {
    SDL_Destroyer sdl_closer;  // Close sdl when destroyed
    SDL_Surface* screen;

    initPaths();

    parse(argc, argv);

    top10::util::global_time.setSlowFactor(slow_factor);

    CleanPtr<FFDevice> clean_ffdev(ffdev);	// delete ffdev is performed when clean_ffdev falls out of scope
    /* Initialize the SDL library */
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK | SDL_INIT_AUDIO) < 0 ) {
      throw Error(string("Couldn't initialize SDL: ") + string(SDL_GetError()));
    }
    // Open all joysticks
    int n_sticks = SDL_NumJoysticks();
    std::vector<SDL_Joystick*> joysticks;
    for (int i=0; i<n_sticks; ++i) {

      SDL_Joystick* joy = SDL_JoystickOpen(i);
      joysticks.push_back(joy);
      if (joy) {
	std::cout<<"Opened joystick "<<i<<std::endl;

	std::cout<<"name: "<<SDL_JoystickName(i)<<" ";
	std::cout<<"Axes: "<<SDL_JoystickNumAxes(joy)<<" ";
	std::cout<<"Buttons: "<<SDL_JoystickNumButtons(joy)<<" ";
	std::cout<<"Balls: "<<SDL_JoystickNumBalls(joy)<<" ";
      }
      else std::cout<<"Failed top open joystick "<<i<<std::endl;
    }

    // Show an uggly splash screen
    //    screen = splash();

    /* Clean up on exit, exit on window close and interrupt */
    sdl_closer.enable(SDL_Destroyer::quit);
  
    // Initialize GL stuff
    screen = initGL(depth, w, h, full_screen);

    // First load the default settings
    std::string kart_path = top10::util::PathFinder::find("default_kart");
    if (kart_path.empty()) throw Error("Could not find default_kart");
    else {
      std::ifstream kart_in(kart_path.c_str());
      if (!kart_in) throw Error("Could not open ")+kart_path;
      top10::physX::Kart::initDefaultParameters(kart_in);
    }
    // Then load the specific settings
    if (!kart_filename.empty()) {
      kart_path = top10::util::PathFinder::find(kart_filename);
      if (kart_path.empty()) throw Error("Could not find ") + kart_filename;
      else {
	std::ifstream kart_in(kart_path.c_str());
	if (!kart_in) throw Error("Could not open ")+kart_path;
	top10::physX::Kart::initDefaultParameters(kart_in);
      }
    }

    // replay/record
    if (record_to_file) {
      rec_out = new std::ofstream(record_filename.c_str());
      if (!*rec_out) throw Error("Could not open output file ") + record_filename;
    }
    if (replay_from_file) {
      rep_in = new std::ifstream(replay_filename.c_str());
      if (!*rep_in) throw Error("Cold not open input file ") + replay_filename;
    }

    // Simulation
    Simulation *sim(0);
    // If we are replaying or benchmarking, create the simulation
    if (replay_from_file || physx_benchmark) {
      top10::track::Track* track = top10::track::loadTrack(track_filename);
      KartGL kart_gl("data/karts/default_model");
      sim = new Simulation(track, rep_in, 0, kart_gl);
      sim->resizeEvent(w, h);
    }

    //    if (ffdev) ffdev->setKart(sim.karts[0]);

    // Init font lib
    fntInit();

    // Audio
    AudioDevice* audio_device(0);
    if (use_audio) {
      audio_device = new AudioDevice;

      /* Open the audio device, forcing the desired format */
      SDL_AudioSpec wanted;
      wanted.freq = 22050;
      wanted.format = AUDIO_S16;
      wanted.channels = 2;    /* 1 = mono, 2 = stereo */
      wanted.samples = 256;  /* Good low-latency value for callback */
      wanted.callback = fill_audio;
      wanted.userdata = (void*)audio_device;
      
      if ( SDL_OpenAudio(&wanted, NULL) < 0 ) {
	throw Error(string("Couldn't open audio: ") + string(SDL_GetError()));
      }
      sdl_closer.enable(SDL_Destroyer::sound);
      cout<<"Audio device opened"<<endl;
    }

    // Create menus
    bool quit;
    ActionMapping actions;
    try {
      ActionMapping tmp_actions;
      if (!top10::util::PathFinder::find("inputs").empty())
	tmp_actions = loadActionMapping("inputs");
      actions = tmp_actions;
    }
    catch (std::string err) {
      std::cerr<<"Could not load action mapping: "<<err<<std::endl;
    }

    UIPanel panel;
    UIPanel::Menu* menu = new MainMenu(&panel, &quit, &actions, &sim, audio_device);
    panel.setFrontMenu(menu);
    panel.resizeEvent(w,h);
    panel.setMenu(menu);

    /* Main loop */
    SDL_Event event;
    Uint32 t_fps = SDL_GetTicks();
    int n_fps = 0;
    for (quit=false; !quit;) {

      // Handle events
      while (!quit && SDL_PollEvent(&event)) {

	if (event.type == SDL_KEYDOWN && (event.key.keysym.sym == SDLK_ESCAPE || panel.getCurrentMenu()))
	  panel.event(event);
	else if (event.type == SDL_QUIT)
	  quit = true;
	else {
	  ActionMapping::const_iterator action = actions.find(event);
	  if (action != actions.end()) {
	    switch(action->second) {
	    case Steering:
	    case Acceleration:
	    case Braking:
	    case AccelerationAndBraking:
	    case BrakingAndAcceleration:
	      if (!replay_from_file && sim)
		sim->sendEvent(action->second, Control::getValue(event));
	      break;

	    case SteerLeft:
	      if (!replay_from_file && sim)
		sim->sendEvent(Steering, action->first.invertedEvent(event)?0:-32767);
	      break;
	    case SteerRight:
	      if (!replay_from_file && sim)
		sim->sendEvent(Steering, action->first.invertedEvent(event)?0:32767);
	      break;
	    case Accelerate:
	      if (!replay_from_file && sim)
		sim->sendEvent(Acceleration, action->first.invertedEvent(event)?-32767:32767);
	      break;
	    case Brake:
	      if (!replay_from_file && sim)
		sim->sendEvent(Braking, action->first.invertedEvent(event)?-32767:32767);
	      break;

	    case NextCamera:
	      if (sim && !action->first.invertedEvent(event))  // We do not want to change the camera on key release
		sim->nextCamera();
	      break;
	    case PrevCamera:
	      if (sim && !action->first.invertedEvent(event))
		sim->previousCamera();
	      break;

	    default:
	      std::cerr<<"Action "<<action->second<<" not handled yet";
	      break;
	    }
	  }
	}
      	
      }

      if (quit) continue;

      // Update world and display it
      if (!physx_benchmark) {
	if (sim) {
	  if (sim->getWet()) glClearColor(gray_sky[0], gray_sky[1], gray_sky[2], gray_sky[3]);
	  else               glClearColor(blue_sky[0], blue_sky[1], blue_sky[2], blue_sky[3]);
	}
	else glClearColor(black_color[0], black_color[1], black_color[2], black_color[3]);

	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	Uint32 t = top10::util::global_time.getTicks();
	if (sim) {
	  sim->update(t);
	  sim->paintGL();
	}
	//	if (ffdev) ffdev->render(t2-t);	

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glScalef(0.05, 0.1, 1);
	glTranslatef(-10.0, 5.0, 0.0);
	glColor4f(1.0f, 1.0f, 0.0f, 0.8f);
	panel.updateGL();
	glEnable(GL_DEPTH_TEST);
	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	
	glFinish();

	// Stats: compute fps
#if 0
	++n_fps;
	Uint32 t2_fps = SDL_GetTicks();
	int dt_fps = t2_fps - t_fps;
	if (dt_fps > 500) {
	  t_fps = t2_fps;
	  std::cout<<"FPS: "<<2*n_fps<<std::endl;
	  n_fps=0;
	}
#endif
      }
      else if (sim) {
	Uint32 T = 60000;
	Uint32 bench0 = SDL_GetTicks();
	sim->update(top10::util::global_time.getTicks()+T);
	Uint32 bench1 = SDL_GetTicks();
	
	cout<<"Physics engine benchmark: "<<(bench1 - bench0)<<" ms to simulate "<<T<<" ms"<<endl;
	cout<<"Ratio: "<<(float)T/(bench1-bench0)<<std::endl;
	quit = true;
      }
    
      SDL_GL_SwapBuffers();
    }
    saveActionMapping(top10::util::UserFile::getPath()+"/inputs", actions);
  }
  catch (const std::string& e) {
    cerr<<"Fatal error caught: "<<e<<endl;
  }
#if 0
  catch (...) {
    cerr<<"Unknown error caught"<<endl;
  }
#endif
 
 if (rep_in) delete rep_in;
 if (rec_out) delete rec_out;
 
 cout<<"Exit"<<endl;
 
 return 0;
}
