/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Simulation.hh"
#include "physX/Wheel.hh"

#include <sstream>
#include <SDL/SDL_image.h>
#include <iomanip>
#include "util/PathFinder.hh"
#include "util/SlowTime.hh"
#include "FakeKart.hh"

using namespace top10::ui_interactive;
using namespace std;

using namespace top10::math;
using namespace top10::racing;
using top10::physX::Wheel;
using top10::physX::World;


Simulation::Simulation(top10::track::Track* _track,
		       istream* in,
		       ostream* out,
		       const KartGL& kgl):
  top10::util::Debug("Simulation", false),
  track(_track),
  isWet(false),
  kart_gl(kgl),
  current_lap_records(1),
  steer_angle(0),
  rep_in(in),
  rec_out(out),
  current_t(0),
  cam_num(1),
  low_speed_col(0.7, 0.7, 0.1),
  hi_speed_col(1, 0.5, 0.4),
  track_mesh(_track->getMesh())
{
  checkpoints = track->getCheckPoints();
  grid = track->getStartingArea();

  // OpenGL stuff
  for (int i=0; i<=100; ++i) {
    speed_gauge.addXY(i/100.0, sqrt(i/100.0));
  }

  {
    Vector pos;
    Vector dir;
    grid->getStartingPos(0, pos, dir);
    karts.push_back(new Kart(track, pos + Vector(0,0.5,0), dir, &kart_gl));
  }

  world = new World;
  world->setOctree(top10::physX::Octree(track->getMesh(), track->getDecorations()));

  initItems();
  ofstream states_strm("states");
  world->dumpTo(states_strm);

  setCam();
  setFont("default.txf");

  timers.push_back(KartTimer(karts[0], checkpoints));
  initial_t = current_t = top10::util::global_time.getTicks();
  ghost_it = best_lap_record.begin();
  next_ghost_it = ghost_it;
}

void Simulation::reinit()
{
  {
    ifstream reinit_strm("states");
    world->loadFrom(reinit_strm);
  }
}

void Simulation::initItems()
{
  karts[0]->getRegistered(*world);
}

void Simulation::setFont(std::string font_name)
{
  std::string font_path = top10::util::PathFinder::find(font_name);
  if (font_path.empty()) throw std::string("Could not load font") + font_name;

  font.load(font_path.c_str());
  texout.setFont(&font);
  texout.setPointSize(1);
}

void Simulation::resizeEvent(int w, int h)
{
  screen_w = w;
  screen_h = h;
}

static inline void glVertex(Vector v)
{
  glVertex3f(v.x, v.y, v.z);
}

void Simulation::paintGL()
{
  setCam();
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(cam.getFOV(), ((GLdouble)screen_w)/screen_h, cam.getNear(), cam.getFar());

  GLdouble ref[4][4];
  cam.getGLMatrix(ref);
  glMatrixMode(GL_MODELVIEW);
  glLoadMatrixd((GLdouble*)ref);

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  // enable fog?
  if (isWet) glEnable(GL_FOG);
  else glDisable(GL_FOG);

  // The track
  // First, draw all opaque objects
  glEnable(GL_ALPHA_TEST);
  glEnable(GL_CULL_FACE);
  glAlphaFunc(GL_EQUAL, 1.0);
  track_mesh.drawGL(cam);
  // Then transparent ones, without modifying the depth buffer
#if 0
  glAlphaFunc(GL_LESS, 1.0);
  glDepthMask(false);
  track_mesh.drawGL(cam);
  glDepthMask(true);
#endif
  glDisable(GL_CULL_FACE);
  glDisable(GL_ALPHA_TEST);

  // Karts
  for (vector<Kart*>::const_iterator p = karts.begin();
       p != karts.end();
       ++p) {
      (*p)->drawGL(cam);
  }

  // The ghost kart
  if (next_ghost_it != best_lap_record.end()) {
    FakeKart ghost(&kart_gl);
    ghost.setState(*ghost_it, *next_ghost_it,
		   current_t - current_lap_records[0].getStartTime());
      ghost.drawGL(cam);
  }
  glDisable(GL_LIGHT0);
  glDisable(GL_LIGHTING);

  //DEBUG: parts of the ground octree
  //  glDisable(GL_DEPTH_TEST);
  cam.drawGL();
#if 0
  std::list<top10::math::AxisAlignedBox> blocks = track_mesh.getOctree()->getBlocks(clip);
  glColor3f(1, 0, 0);
  for (std::list<top10::math::AxisAlignedBox>::const_iterator block = blocks.begin();
       block != blocks.end();
       ++block) {
    Vector vertices[8];
    block->getVertices(vertices);
    
    glBegin(GL_LINE_STRIP);
    glVertex(vertices[0]);
    glVertex(vertices[1]);
    glVertex(vertices[3]);
    glVertex(vertices[2]);
    glVertex(vertices[6]);
    glVertex(vertices[4]);
    glVertex(vertices[5]);
    glVertex(vertices[7]);
    glVertex(vertices[3]);
    glEnd();
    
    glBegin(GL_LINES);
    glVertex(vertices[0]);
    glVertex(vertices[2]);
    
    glVertex(vertices[6]);
    glVertex(vertices[7]);
    
    glVertex(vertices[1]);
    glVertex(vertices[5]);
    
    glVertex(vertices[0]);
    glVertex(vertices[4]);
    glEnd();
  }
#endif

  // 2D drawings
  GLenum glerr = glGetError() ;
  if (glerr) cerr<<"Error before"<<gluErrorString(glerr)<<endl;

  glDisable(GL_LIGHT0);
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glScalef(2.0/screen_w, 2.0/screen_h, 1.0);
  glTranslatef(-0.5*screen_w, -0.5*screen_h, 0);

  // Speed Gauge
  glScalef(100, 50, 0);

  // Black background
  glColor3f(0, 0, 0);
  glBegin(GL_QUAD_STRIP);
  for (int i=0; i<=20; i++) {
    glVertex2f(i/20.0, speed_gauge.getY(i/20.0));
    glVertex2f(i/20.0, speed_gauge.getY(i/20.0) + 0.2);
  }
  glEnd();

  // Value
  double spd = karts[0]->getRadPerSec() / karts[0]->getMaxRadPerSec();
  if (spd > 1.0) spd = 1.0;
  glBegin(GL_QUAD_STRIP);
  for (double x=0.0; x<=spd; x+=0.01) {
    Vector color = (1-x)*low_speed_col + x*hi_speed_col;
    glColor3f(color.x, color.y, color.z);
    glVertex2f(x, speed_gauge.getY(x));
    glVertex2f(x, speed_gauge.getY(x) + 0.2);
  }
  glEnd();

  // Print time
  glLoadIdentity();
  glScalef(0.04, 0.1, 1.0);
  glTranslatef(10.0, -9.0, 0.0);

  glColor3f(1.0, 1.0, 1.0);

  unsigned int ms = timers[0].getBestTime();
  unsigned int min = ms/60000;
  unsigned int sec = (ms/1000)%60;
  unsigned int msec = ms%1000;

  texout.begin();
  {
    ostringstream buff;
    buff.fill('0');
    buff<<"Best "<<min<<":"<<setw(2)<<sec<<"."<<setw(3)<<msec;
    texout.start2f(0.0, 0.0);
    texout.puts(buff.str().c_str());
  }

  ms = timers[0].getLastTime();
  min = ms/60000;
  sec = (ms/1000)%60;
  msec = ms%1000;
  {
    ostringstream buff;
    buff.fill('0');
    buff<<"Last "<<min<<":"<<setw(2)<<sec<<"."<<setw(3)<<msec;
    texout.start2f(0.0, 1.5);
    texout.puts(buff.str().c_str());
  }

  ms = timers[0].getCurrentTime();
  min = ms/60000;
  sec = (ms/1000)%60;
  msec = ms%1000;
  {
    ostringstream buff;
    buff.fill('0');
    buff<<"Current "<<min<<":"<<setw(2)<<sec<<"."<<setw(3)<<msec;
    texout.start2f(0.0, 3.0);
    texout.puts(buff.str().c_str());
  }

  texout.end();

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
}

void Simulation::update(unsigned int t_end)
{
  for (; current_t<t_end;
       current_t+=World::time_slice) {
    // read events from file
    if (rep_in) {
      Uint32 e_time;
      
      do {
	Event ev;
      	(*rep_in)>>ev;
	ev.timestamp += initial_t;
	e_time = ev.timestamp;
      	if (!rep_in->eof()) events_to_replay.push(ev);
      } while(!rep_in->eof() && e_time <= current_t);
    }
    // replay events
    if (events_to_replay.size() > 0) {
      do {
	Event ev = events_to_replay.front();
	Uint32 e_time = ev.timestamp;
	if (e_time > current_t) break;
      
	assert (e_time == current_t);

	sendEvent(ev.action, ev.value);
	events_to_replay.pop();
      } while (events_to_replay.size()>0);
    }

    world->simulate();
    
    karts[0]->steer(steer_angle);
    karts[0]->accel(throttle);
    karts[0]->brake(braking);

    timers[0].update(World::time_slice);
    //TODO: next line should be done outside the loop
    karts[0]->updateUI(World::time_slice/1000.0);

    // Update the ghost state iterators
    double relative_t = current_t - current_lap_records[0].getStartTime(); 
    while (next_ghost_it != best_lap_record.end() && 
	   relative_t >= next_ghost_it->timestamp) {
      ++ghost_it;
      ++next_ghost_it;
    }
    // If we just completed a lap...
    if (timers[0].justFinishedLap()) {
      std::cerr<<"Lap just finished"<<std::endl;

      // add the final state to the lap record
      current_lap_records[0].addState(current_t, karts[0]->getKartState());

      // If that's the best lap (or the first), store it
      if (best_lap_record.getTime() == 0 ||
	  timers[0].getLastTime() < best_lap_record.getTime()) {
	std::cerr<<"New best lap"<<std::endl;
	best_lap_record = current_lap_records[0];
	std::cerr<<"best_lap_record.size() "<<best_lap_record.size()<<std::endl;
      }

      // Reset the current lap record
      current_lap_records[0].clear(current_t);

      // Reset the ghost pointers
      ghost_it = best_lap_record.begin();
      assert(ghost_it != best_lap_record.end());
      next_ghost_it = ghost_it +1;
      assert(next_ghost_it != best_lap_record.end());
    }
    // If this is the start of the first lap
    else if (timers[0].justStartedFirstLap()) {
      std::cerr<<"First lap started"<<std::endl;
      // Reset the current lap record
      current_lap_records[0].clear(current_t);
      current_lap_records[0].addState(current_t, karts[0]->getKartState());
    }
    // We store the state every 0.1s
    else if (timers[0].timerRunning() &&
	     (current_lap_records[0].size() == 0 ||
	     current_t - current_lap_records[0].getTime() >= 100)) {
      current_lap_records[0].addState(current_t, karts[0]->getKartState());
    }
  }
}


void Simulation::nextCamera()
{
  if (cam_num == 5) cam_num=0;
  else cam_num++;
  
  setCam();
}

void Simulation::setCam()
{
  switch (cam_num) {
  case 0: cam = karts[0]->getCamera(Kart::In); break;
  case 1: cam = karts[0]->getCamera(Kart::Back); break;
  case 2: cam = karts[0]->getCamera(Kart::Fore); break;
  case 3: cam = karts[0]->getCamera(Kart::Left); break;
  case 4: cam = karts[0]->getCamera(Kart::Right); break;
  case 5: cam = karts[0]->getCamera(Kart::Top); break;
  default: cerr<<"Simulation::setCam(): bad cam_num "<<cam_num<<endl; abort();
  }

  cam.setRatio((double)screen_w/screen_h);
  cam.update();
}

void Simulation::previousCamera()
{
  if (cam_num == 0) cam_num=5;
  else cam_num--;

  setCam();
}

void Simulation::sendEvent(Action action, Sint16 value)
{
  if (rec_out) {
    Event ev;
    ev.timestamp = current_t - initial_t;
    ev.action = action;
    ev.value = value;
    (*rec_out)<<ev<<endl;
  }

  switch (action) {
  case Steering:
    steer_angle = -value/32768.0;
    break;
  case Acceleration:
    throttle = (value+32768)/65536.0;
    break;
  case Braking:
    braking = (value+32768)/65536.0;
    break;
  case AccelerationAndBraking:
    if (value > 0) {
      throttle = value/32768.0;
      braking = 0.0;
    }
    else {
      throttle = 0.0;
      braking = -value/32768.0;
    }
    break;
  case BrakingAndAcceleration:
    if (value < 0) {
      throttle = -value/32768.0;
      braking = 0.0;
    }
    else {
      throttle = 0.0;
      braking = value/32768.0;
    }
    break;

  default: abort();  /* We should not receive other kinds of events */
  }
}

Simulation::~Simulation()
{
  delete world; // Scary, isn't it ?

  for (vector<Kart*>::iterator p=karts.begin(); p!= karts.end(); ++p) {
  	delete *p;
  }
}

#include "helpers/GenericOctree-template.cpp"
