/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_UI_KART_HH
#define TOP10_UI_KART_HH

#include <GL/gl.h>
#include <iostream>

#include "physX/Kart.hh"
#include "Triangle.hh"
#include "Frustum.hh"

#include "racing/LapRecord.hh"

namespace top10 {
  namespace ui_interactive {

    using top10::math::Vector;

    //! Class used to draw a kart using openGL
    class KartGL {
    public:
      KartGL(std::string model_filename);

      //! Issue the opengl commands to draw the kart
      /*! Parameters: state of the kart. */
      void drawGL(Vector translation, Vector mass_center_L,
		  top10::math::Matrix3 orient, double steer_angle,
		  double steer_fl, double steer_fr,
		  double h_fl, double h_fr,
		  double h_rl, double h_rr) const;

      void drawGL(Vector translation, Vector mass_center_L,
		  top10::math::Matrix3 orient, double steer_angle,
		  double steer_fl, double steer_fr,
		  double h_fl, double h_fr,
		  double h_rl, double h_rr,
		  const Frustum& clip) const;

    private:
      static void parsePart(std::istream&, const char* keyword,
			    std::string* alias,
			    std::string* object, top10::math::Matrix4* M);

      //! Indexes in the transforms and models arrays
      static const int BODY = 0;
      static const int WHEEL_FL = 1;
      static const int WHEEL_FR = 2;
      static const int WHEEL_RL = 3;
      static const int WHEEL_RR = 4;
      static const int STEER = 5;

      //! All 3d models that are part of the kart
      std::map<std::string, TriangleSet*> model_map;

      //! The transformation to apply to each part
      top10::math::Matrix4 transforms[6];

      //! Models for each part
      TriangleSet* models[6];
      int mesh_idxs[6];

      //! The center of the steering wheel
      top10::math::Vector steer_center;
      //! The axis of the steering wheel
      top10::math::Vector steer_axis;

      //! The attachment point for each wheel
      top10::math::Vector fl, fr, rl, rr;

      top10::math::Box bounding_box;
    };

    class Kart: public top10::physX::Kart
    {
    public:
      enum camEnum {
	Back,
	In,
	Fore,
	Left,
	Right,
        Top
      };

    public:
      /**
	 Creates a new kart.
	 \param starting_pos Where to place the kart
	 \param orientation The initial orientation of the kart
      */
      Kart(top10::track::Track* track, Vector starting_pos=Vector(0, 0, 0), Vector orientation=Vector(-1, 0, 0), const KartGL* kart_gl=0);

      /**
	 Update the kart.
	 This member does not update the physics related data. Instead, it
	 accumulates data useful to generate sounds
      */
      void updateUI(double dt);

      /**
	 Gets the amount of friction for fore and back wheels on the left side.
	 It can be used for sound producing. It is not intended to be used for
	 computations.
      */
      double getLeftFriction() const;

      /**
	 Gets the amount of friction for fore and back wheels on the right side.
	 It can be used for sound producing. It is not intended to be used for
	 computations.
      */
      double getRightFriction() const;


      //! Get the load on the left side
      double getLeftLoad() const;
      double getRightLoad() const;
      
      /**
         Gets the average force applied to the sides
      */
      inline double getSideForce() const {return side_force_accum;}
       
      /**
	 Issues OpenGL instructions drawing the kart.
      */
      void drawGL(const Frustum&) const;
      void drawGL() const;

      top10::racing::KartState getKartState() const;

      Frustum getCamera(camEnum) const;

    private:
      static const double memory_factor;
      double friction_left_accum;
      double friction_right_accum;
      double friction_left_diff;
      double friction_right_diff;
      double load_left_accum;
      double load_right_accum;
      double side_force_accum;
      Vector friction_force_fl;
      Vector friction_force_fr;
      Vector friction_force_rl;
      Vector friction_force_rr;

      const KartGL* kart_gl;
    };
  };
};

#endif
