/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <cstdio>
#include <errno.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <unistd.h>

#include "ForceFeedback.hh"

#define BITS_PER_LONG (sizeof(long) * 8)
#define OFF(x)  ((x)%BITS_PER_LONG)
#define BIT(x)  (1UL<<OFF(x))
#define LONG(x) ((x)/BITS_PER_LONG)
#define test_bit(bit, array)    ((array[LONG(bit)] >> OFF(bit)) & 1)

using namespace std;

top10::ui_interactive::FFJoystick::FFJoystick(int devn) throw(top10::util::Error):
	fd(-1)
{
  using top10::util::Error;

  // Simulate a kind of centrifugal force on the user
  centrifugal_magnitude.addXY(0, 0);
  centrifugal_magnitude.addXY(100, 0);
  centrifugal_magnitude.addXY(5000, 1);

#ifdef HAVE_FF
  char dev_name[128];

  sprintf(dev_name, "/dev/input/event%d", devn);

  // Open force feedback /dev entry
  fd = open(dev_name, O_RDWR);
  if (fd == -1) throw Error(string("FFJoystick constructor:")+sys_errlist[errno]);

  // Check if the device is capable of rendering force effects
  int nlongs = FF_MAX/(8*sizeof(unsigned long)) +1;
  unsigned long features[nlongs];
  if (ioctl(fd, EVIOCGBIT(EV_FF, sizeof(unsigned long) * nlongs), features) == -1) {
    close(fd);
    throw Error(string("FFJoystick constructor:")+sys_errlist[errno]);
  }
  if (!( test_bit(FF_CONSTANT, features) && test_bit(FF_PERIODIC, features) &&
	 test_bit(FF_SPRING, features) && test_bit(FF_DAMPER, features) )) {
    close(fd);
    throw Error("FFJoystick constructor: device does not provide all required effects");
  }

  // Check if the device can play all effects at the same time
  int n_dev_effects;
  if (ioctl(fd, EVIOCGEFFECTS, &n_dev_effects) == -1) {
    close(fd);
    throw Error(string("FFJoystick constructor:")+sys_errlist[errno]);
  }
  if (n_dev_effects < n_effects) {
    close(fd);
    throw Error("FFJoystick constructor: device can not play enough effects");
  }

  // Initialise effects structures

  // Vibrations due to bumps
  bumps.type = FF_PERIODIC;
  bumps.id = -1;
  bumps.u.periodic.waveform = FF_TRIANGLE;
  bumps.u.periodic.period = 0;	// To be updated by render()
  bumps.u.periodic.magnitude = 0;	// To be updated by render()
  bumps.u.periodic.offset = 0;
  bumps.u.periodic.phase = 0;
  bumps.direction = 0x4000;	// X axis
  bumps.u.periodic.envelope.attack_length = 0;
  bumps.u.periodic.envelope.attack_level = 0;
  bumps.u.periodic.envelope.fade_length = 0;
  bumps.u.periodic.envelope.fade_level = 0;
  bumps.trigger.button = 0;
  bumps.trigger.interval = 0;
  bumps.replay.length = 0xFFFF;	// Never stop
  bumps.replay.delay = 0;

  // centrifugal force
  centrifugal.type = FF_CONSTANT;
  centrifugal.id = -1;
  centrifugal.u.constant.level = 0;	// To be updated by render()
  centrifugal.direction = 0x4000; // To be updated by render()
  centrifugal.u.constant.envelope.attack_length = 0;
  centrifugal.u.constant.envelope.attack_level = 0;
  centrifugal.u.constant.envelope.fade_length = 0;
  centrifugal.u.constant.envelope.fade_level = 0;
  centrifugal.trigger.button = 0;
  centrifugal.trigger.interval = 0;
  centrifugal.replay.length = 0x0100;  /* 1 second */
  centrifugal.replay.delay = 0;

  centrifugal_wait_time = 100000;

  ///\todo Friction of steering wheel

  ///\todo Make the wheel come to its center position

#else // HAVE_FF
  throw Error("FFJoystick constructor: Force feedback not enabled at compilation time");
#endif
}

void top10::ui_interactive::FFJoystick::tellEffectToggled(FFDevice::effect fx)
{
#ifdef HAVE_FF
  // Is this effect going to be deactivated ?
  if (enabled_effects & fx) {
    // switch it off
    struct input_event stop;
    stop.type = EV_FF;
    stop.value = 0;
    switch (fx) {
    case Bumps: stop.code = bumps.id; bumps_playing = false; break;
    case Centrifugal: stop.code = centrifugal.id; break;
    case Steer_drag: stop.code = steer_friction.id; break;
    case Steer_center: stop.code = center_back.id; break;
    }

    write(fd, (const void*) &stop, sizeof(stop));

  }
  else {
     // Play it
    struct input_event play;

    play.type = EV_FF;
    play.value = 1;
    switch (fx) {
    case Bumps: play.code = bumps.id; bumps_playing = true; break;
    case Centrifugal: play.code = centrifugal.id; break;
    case Steer_drag: play.code = steer_friction.id; break;
    case Steer_center: play.code = center_back.id; break;
    }
    
    write(fd, (const void*) &play, sizeof(play));
  }
#endif
}

void top10::ui_interactive::FFJoystick::render(int ms_ellapsed)
{
#ifdef HAVE_FF
  // Bumps
  if (enabled_effects & Bumps) {
    double v = kart->getSpeed().size();
    if (v > 1) {
      const double bump_size = 2;
      bumps.u.periodic.period = __u16(1000 * bump_size / v);
      bumps.u.periodic.magnitude = __s16(0x7fff * kart->getBumps());

      std::cout<<"period "<<bumps.u.periodic.period<<std::endl;
      std::cout<<"magnitude "<<bumps.u.periodic.magnitude<<std::endl;
      std::cout<<"bumps "<<kart->getBumps()<<std::endl;
   
      // Download updated effect
      ioctl(fd, EVIOCSFF, &bumps);
      
      if (!bumps_playing) {
        bumps_playing = true;

        struct input_event play;
        play.type = EV_FF;
        play.code = bumps.id;
        play.value = 1;   
      
        write(fd, (const void*) &play, sizeof(play));
      }	
    }
    else if (bumps_playing) {
      struct input_event stop;
      stop.type = EV_FF;
      stop.code = bumps.id;
      stop.value = 0;   
      
      write(fd, (const void*) &stop, sizeof(stop));
      
      bumps_playing = false;
    }
  }

  // Centrifugal force
  centrifugal_wait_time += ms_ellapsed;
  if ((enabled_effects & Centrifugal) && (centrifugal_wait_time > centrifugal_update_time)) {

    centrifugal.u.constant.level = -__s16(0x7fff * centrifugal_magnitude.getY(fabs(kart->getSideForce())));
    if (kart->getSideForce() < 0) centrifugal.u.constant.level = -centrifugal.u.constant.level;
    cout<<"side_force ="<<kart->getSideForce()<<endl;

    // Download updated effect
    ioctl(fd, EVIOCSFF, &centrifugal);
  }
#endif
}

top10::ui_interactive::FFJoystick::~FFJoystick()
{
  cout<<"FFJoystick destroyed"<<endl;

#ifdef HAVE_FF
  // Close device
  close(fd);
#endif
}

top10::ui_interactive::FFDevice*
top10::ui_interactive::ff_factory(const string& devname, int n)
{
	if (devname == "ifw" || devname == "iforce wheel")
		return new FFJoystick(n);
	return 0;
}
