/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Controls.hh"
#include "util/PathFinder.hh"
#include <fstream>

using namespace top10::ui_interactive;

bool Control::matchEvent(SDL_Event e) const
{
  /* Digital events */
  if (e.type == SDL_KEYUP || e.type == SDL_KEYDOWN)
    return event.type == SDL_KEYDOWN && event.key.keysym.sym == e.key.keysym.sym;

  else if (e.type == SDL_MOUSEBUTTONDOWN || e.type == SDL_MOUSEBUTTONUP)
    return event.type == SDL_MOUSEBUTTONDOWN && event.button.button == e.button.button;

  else if (e.type == SDL_JOYBUTTONDOWN || e.type == SDL_JOYBUTTONUP)
    return event.type == SDL_JOYBUTTONDOWN && e.jbutton.which == event.jbutton.which &&
      e.jbutton.button == event.jbutton.button;
  
  /* Analog events */
  else if (e.type == SDL_JOYAXISMOTION)
    return e.jaxis.which == event.jaxis.which && e.jaxis.axis == event.jaxis.axis;
  
  return false;
}

bool Control::invertedEvent(SDL_Event e) const
{
  assert(matchEvent(e));
  return e.type != event.type;
}

bool Control::operator==(Control other) const
{
  if (accepted(other.event)) return matchEvent(other.event);
  return false;
}

Sint16 Control::getValue(SDL_Event ev)
{
  switch (ev.type) {
  case SDL_JOYAXISMOTION:
    return ev.jaxis.value;
  case SDL_KEYDOWN:
  case SDL_MOUSEBUTTONDOWN:
  case SDL_JOYBUTTONDOWN:
    return 32767;
  case SDL_KEYUP:
  case SDL_MOUSEBUTTONUP:
  case SDL_JOYBUTTONUP:
    return 0;

  default:
    abort();
  }
}

std::istream& top10::ui_interactive::operator>>(std::istream& in, Action& action)
{
  std::string token;
  in>>token;
  
  if (token == "Steering") action = Steering;
  else if (token == "Acceleration") action = Acceleration;
  else if (token == "Braking") action = Braking;
  else if (token == "AccelerationAndBraking") action = AccelerationAndBraking;
  else if (token == "BrakingAndAcceleration") action = BrakingAndAcceleration;

  else if (token == "SteerLeft") action = SteerLeft;
  else if (token == "SteerRight") action = SteerRight;
  else if (token == "Accelerate") action = Accelerate;
  else if (token == "Brake") action = Brake;
  else if (token == "NextCamera") action = NextCamera;
  else if (token == "PrevCamera") action = PrevCamera;
  else if (token == "NextKart") action = NextKart;
  else if (token == "PrevKart") action = PrevKart;
  //  else throw std::string("Invalid DigitalAction: ")+token;

  return in;
}

std::ostream& top10::ui_interactive::operator<<(std::ostream& out, Action action)
{
#define CASEOUT(e, t) case e: out<<t; break

  switch (action) {
    CASEOUT(Steering, "Steering");
    CASEOUT(Acceleration, "Acceleration");
    CASEOUT(Braking, "Braking");
    CASEOUT(AccelerationAndBraking, "AccelerationAndBraking");
    CASEOUT(BrakingAndAcceleration, "BrakingAndAcceleration");

    CASEOUT(SteerLeft, "SteerLeft");
    CASEOUT(SteerRight, "SteerRight");
    CASEOUT(Accelerate, "Accelerate");
    CASEOUT(Brake, "Brake");
    CASEOUT(NextCamera, "NextCamera");
    CASEOUT(PrevCamera, "PrevCamera");
    CASEOUT(NextKart, "NextKart");
    CASEOUT(PrevKart, "PrevKart");

  default: assert(0);
  }

  return out;
#undef CASEOUT
}

std::istream& top10::ui_interactive::operator>>(std::istream& in, Control& ctrl)
{
  std::string token;
  SDL_Event event;

  in>>token;
  if (token == "MouseButton") {
    event.type = SDL_MOUSEBUTTONDOWN;

    int n;
    in>>n;
    if (n<0 || n>255) throw std::string("MouseButton number out of range");
    event.button.button = n;
  }
  else if (token == "JoyButton") {
    event.type = SDL_JOYBUTTONDOWN;

    int which, but;
    in>>which>>but;
    if (which<0 || which>255) throw std::string("Joy number out of range");
    if (but<0 || but>255) throw std::string("Joy button out of range");
    event.jbutton.which = which;
    event.jbutton.button = but;
  }
  else if (token == "KeyPress") {
    event.type = SDL_KEYDOWN;

    int n;
    in>>n;
    if (n<0) throw std::string("Key out of range");
    event.key.keysym.sym = (SDLKey)n;  // Yuck!
  }
  else if (token == "JoyAxis") {
    int which, axis;
    in>>which>>axis;
    event.type = SDL_JOYAXISMOTION;
    event.jaxis.which = which;
    event.jaxis.axis = axis;
  }
  //  else throw std::string("Bad type of digital action: ")+token;

  ctrl = Control(event);

  return in;
}

void Control::output(std::ostream& out) const
{
  switch (event.type) {
  case SDL_MOUSEBUTTONDOWN:
    out<<"MouseButton "<<(int)event.button.button;
    break;

  case SDL_JOYBUTTONDOWN:
    out<<"JoyButton "<<(int)event.jbutton.which<<" "<<(int)event.jbutton.button;
    break;

  case SDL_KEYDOWN:
    out<<"KeyPress "<<(int)event.key.keysym.sym;
    break;

  case SDL_JOYAXISMOTION:
    out<<"JoyAxis "<<(int)event.jaxis.which<<" "<<(int)event.jaxis.axis;
    break;

  default: abort();
  }
}

bool Control::accepted(SDL_Event e)
{
  switch (e.type) {
  case SDL_KEYDOWN:
  case SDL_MOUSEBUTTONDOWN:
  case SDL_JOYBUTTONDOWN:
  case SDL_JOYAXISMOTION:
    return true;
  default:
    return false;
  }
  return false;
}

ActionMapping top10::ui_interactive::loadActionMapping(std::string filename)
{
  std::string path = top10::util::PathFinder::find(filename);
  if (path.empty()) throw std::string("Could not find ")+filename;

  std::ifstream in(path.c_str());
  if (!in) throw std::string("Could not open ")+filename;

  ActionMapping digs;
  while(!in.eof()) {
    Control ctrl;
    Action action;
    in>>ctrl>>action;
    if (!in.eof())
      digs.insert(ctrl, action);
  }

  return digs;
}

void top10::ui_interactive::saveActionMapping(std::string filename, ActionMapping digs)
{
  std::ofstream out(filename.c_str());
  if (!out) throw std::string("Could not open ")+filename+" for writing";

  for (ActionMapping::const_iterator it = digs.begin();
       it != digs.end();
       ++it) {
    out<<(it->first)<<" "<<(it->second)<<std::endl;
  }
}

ActionMapping::ActionMapping()
{}

ActionMapping::const_iterator ActionMapping::find(SDL_Event e) const
{
  const_iterator f;
  for (f = l.begin(); f != l.end() && !f->first.matchEvent(e); ++f);
  return f;
}

ActionMapping::const_iterator ActionMapping::find(Control ctrl) const
{
  const_iterator f;
  for (f = l.begin(); f != l.end() && !(f->first == ctrl); ++f);
  return f;  
}

void ActionMapping::insert(Control ctrl, Action action)
{
  // if there is already a confliction entry with action ...
  std::list<std::pair<Control, Action> > tmp_l;
  for (iterator it = l.begin(); it != l.end(); ++it) {
    // Do not keep it
    if (it->second != action) tmp_l.push_back(*it);
  }
  l = tmp_l;

  // if there is an entry with ctrl...
  iterator f;
  for (f = l.begin(); f != l.end() && !(f->first == ctrl); ++f);
  if (f == l.end()) l.push_back(std::make_pair(ctrl, action));
  else f->second = action;
}
