/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "SurfacesEditor.hh"

using namespace top10::tracked;

// Road, DustyRoad, Border, Grass, Sand, Dirt
const unsigned char SurfacesEditor::r[6] = {160, 100, 255,   0, 255, 127};
const unsigned char SurfacesEditor::g[6] = {160, 100,   0, 255, 255,   0};
const unsigned char SurfacesEditor::b[6] = {160, 100,   0,   0,   0,   0};

SurfacesEditor::SurfacesEditor():
  mesh_ed(0)
{
  surfaces[Road] =      &road_surfaces;
  surfaces[DustyRoad] = &dusty_road_surfaces;
  surfaces[Border] =    &border_surfaces;
  surfaces[Grass] =     &grass_surfaces;
  surfaces[Sand] =      &sand_surfaces;
  surfaces[Dirt] =      &dirt_surfaces;
  reset();
}

void SurfacesEditor::reset()
{
  current_surface_name.clear();
  current_surface_type = Road;
  if (mesh_ed) setAlternateColors();
}

std::set<std::string> SurfacesEditor::getCurrentSurfaceSet() const
{
  switch (current_surface_type) {
  case Road:      return road_surfaces;       break;
  case DustyRoad: return dusty_road_surfaces; break;
  case Grass:     return grass_surfaces;      break;
  case Border:    return border_surfaces;     break;
  case Sand:      return sand_surfaces;       break;
  case Dirt:      return dirt_surfaces;       break;
  }
  abort();
}

void SurfacesEditor::pick()
{
  assert(mesh_ed);

  current_surface_name = mesh_ed->getPickedSurface();
}

void SurfacesEditor::add()
{
  if (current_surface_name.empty()) throw std::string("No surface name to add to the current surface type");

  int found=0;
  // 6 surface types... (ugh)
  for (int i=0; i<6; ++i) {
    if (i==current_surface_type) surfaces[i]->insert(current_surface_name);
    else {
      std::set<std::string>::const_iterator f=surfaces[i]->find(current_surface_name);
      if (f != surfaces[i]->end()) {
	surfaces[i]->erase(f);
	found++;
      }
    }
  }
  assert(found==0 || found==1);

  // Update the alternate color info of the affected surface
  assert(mesh_ed);
  top10::ui_interactive::TriangleSet* gl_polys = mesh_ed->getGLPolys();
  top10::helpers::PolygonSet* polys = mesh_ed->getPolys();
  assert(gl_polys); assert(polys);
  for (int i=(int)polys->surfaces.size()-1; i>=0; --i) {
    if (polys->surfaces[i].name == current_surface_name) {
      gl_polys->textures[i].r = r[(int)current_surface_type];
      gl_polys->textures[i].g = g[(int)current_surface_type];
      gl_polys->textures[i].b = b[(int)current_surface_type];
      break;
    }
  }
}

void SurfacesEditor::remove()
{
  if (current_surface_name.empty()) throw std::string("No surface name to remove");

  int found=0;
  // 6 surface types... (ugh)
  for (int i=0; i<6; ++i) {
    std::set<std::string>::const_iterator f=surfaces[i]->find(current_surface_name);
    if (f != surfaces[i]->end()) {
      surfaces[i]->erase(f);
      found++;
    }
  }
  assert(found==0 || found==1);

  // Update the alternate color info of the affected surface
  assert(mesh_ed);
  top10::ui_interactive::TriangleSet* gl_polys = mesh_ed->getGLPolys();
  top10::helpers::PolygonSet* polys = mesh_ed->getPolys();
  assert(gl_polys); assert(polys);
  for (int i=(int)polys->surfaces.size()-1; i>=0; --i) {
    if (polys->surfaces[i].name == current_surface_name) {
      gl_polys->textures[i].r = 0;
      gl_polys->textures[i].g = 0;
      gl_polys->textures[i].b = 0;
      break;
    }
  }
}

void SurfacesEditor::setAlternateColors()
{
  assert(mesh_ed);

  top10::ui_interactive::TriangleSet* gl_polys = mesh_ed->getGLPolys();
  top10::helpers::PolygonSet* polys = mesh_ed->getPolys();
  assert(gl_polys); assert(polys);

  for (int i=(int)polys->surfaces.size()-1; i>=0; --i) {
    bool found=false;
    for (int surf_idx=0; !found && surf_idx<6; ++surf_idx) {
      if (surfaces[surf_idx]->find(polys->surfaces[i].name) != surfaces[surf_idx]->end()) {
	found = true;
	gl_polys->textures[i].r = r[surf_idx];
	gl_polys->textures[i].g = g[surf_idx];
	gl_polys->textures[i].b = b[surf_idx];
      }
    }
    if (!found) {
      gl_polys->textures[i].r = 0;
      gl_polys->textures[i].g = 0;
      gl_polys->textures[i].b = 0;
    }
  }
}

void SurfacesEditor::drawGL_impl() const
{
  assert(mesh_ed);
  top10::ui_interactive::TriangleSet* gl_polys = mesh_ed->getGLPolys();
  if (!gl_polys) return;

  top10::helpers::PolygonSet* polys = mesh_ed->getPolys();
  assert(polys);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  mesh_ed->setTransformGL();

  // Draw all meshes
  for (int i=(int)polys->meshes.size()-1; i>=0; --i) {
    gl_polys->drawPartGL(i, false, 0.5, true); // No wireframe, alpha 0.5, use alternate color
  }

  glPopMatrix();
}
