/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_LAPRECORD_HH
#define TOP10_LAPRECORD_HH

namespace top10 {
  namespace racing {
    struct KartState {
      unsigned int timestamp;
      top10::math::Matrix3 orient;
      top10::math::Vector translation;
      double steer;
    };

    //! Record a full lap
    class LapRecord {
    public:
      typedef std::vector<KartState> ContainerT;
      typedef ContainerT::const_iterator const_iterator;

      LapRecord(): lap_start(0) {}

      /*!
	\note s.timestamp is absolute, but the stored value is relative to the
	start of this lap.
      */
      inline void addState(unsigned int ts, KartState s) {
	assert(lap_start && ts >= lap_start);
	s.timestamp = ts-lap_start;
	states.push_back(s);
      }
      inline const_iterator begin() const { return states.begin(); }
      inline const_iterator end() const { return states.end(); }

      inline unsigned int getTime() {
	if (states.empty()) return 0;
	else return states.rbegin()->timestamp;
      }

      inline unsigned int getStartTime() { return lap_start; }

      inline void swap(LapRecord& other) {
	states.swap(other.states);
	unsigned int tmp = lap_start;
	lap_start = other.lap_start;
	other.lap_start = tmp;
      }

      inline void clear(unsigned int start) {
	states.clear();
	lap_start = start;
      }

      inline int size() const {
	return states.size();
      }

    private:
      ContainerT states;
      //! Timestamp of the start of the lap
      unsigned int lap_start;
    };

  };
};

#endif
