/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_KART_TIMER_HH
#define TOP10_KART_TIMER_HH

#include <vector>

#include "physX/Kart.hh"
#include "CheckPoint.hh"

namespace top10 {
  namespace racing {

	using top10::physX::Kart;

    /**
       This class keeps track of the lap times.
    */
    class KartTimer
    {
    public:
      /**
	 \param kart a pointer to the kart being tracked
	 \param checkpoints a pointer to the list of checkpoints
      */
      KartTimer(const Kart* kart, const CheckPoints* checkpoints);

      /**
	 Update the times and checks if a checkpoint was passed.
	 Needs to be called every simulation step
      */
      void update(unsigned int ms_elapsed);

      inline const std::vector<unsigned int>& getLapTimes() const {return lap_times;}

      inline unsigned int getBestTime() const {
	return lap_times.empty()? 0 : best_time;
      }

      inline unsigned int getCurrentTime() const {
	return starting_line_crossed? (current_ms - last_ms) : 0;
      }

      inline unsigned int getLastTime() const {
	return lap_times.empty()? 0 : *(lap_times.rbegin());
      }

      inline bool justFinishedLap() const {
	return !lap_times.empty() && just_crossed_line;
      }
      inline bool justStartedFirstLap() const {
	return lap_times.empty() && just_crossed_line;
      }
      inline bool timerRunning() const {
	return starting_line_crossed;
      }

    private:
      const Kart* kart;
      const CheckPoints* checkpoints;
      CheckPoints::const_iterator next_checkpoint;

      std::vector<unsigned int> lap_times;
      unsigned int best_time;

      /// time at which the starting line was last crossed
      unsigned int last_ms;

      /// Current time
      unsigned int current_ms;

      Vector current_pos, old_pos;
      bool first_time;
      bool starting_line_crossed;
      bool just_crossed_line;

      //DEBUG
      friend class top10::ui_interactive::Simulation;
    };
  };
};

#endif
