/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "CheckPoint.hh"

using namespace top10::racing;
using namespace std;

CheckPoint::CheckPoint(double x1, double z1, double x2, double z2)
{
  p1.x = x1;
  p1.z = z1;
  p2.x = x2;
  p2.z = z2;
  plane = Plane(p1, p2, p1+Vector(0, 1, 0));
}

CheckPoint::CheckPoint(const Vector& a, const Vector& b)
{
  p1.x = a.x;
  p1.z = a.z;
  p2.x = b.x;
  p2.z = b.z;
  plane = Plane(p1, p2, p1+Vector(0, 1, 0));  
}

bool CheckPoint::checkPassed(const Vector& old_pos, const Vector& new_pos) const
{
  double v1=(old_pos-p1)*plane.normal;
  double v2=(new_pos-p1)*plane.normal;

  if (v1<0 && v2<0) return false;
  if (v1>0 && v2>0) return false;

  Vector w=new_pos-old_pos;
  Vector u1 = p1-old_pos;
  Vector u2 = p2-old_pos;
  if ((u1.x*u2.x +  u1.z*u2.z)> 0) return false;

  return true;
}

CheckPoint& CheckPoint::operator=(const CheckPoint& other)
{
  p1 = other.p1;
  p2 = other.p2;
  plane = other.plane;
  return *this;
}

istream& top10::racing::operator>>(istream& in, CheckPoint &cp)
{
  double x1, z1, x2, z2;
  in>>x1>>z1>>x2>>z2;
  cp = CheckPoint(x1, z1, x2, z2);
  return in;
}

ostream& top10::racing::operator<<(ostream& out, const CheckPoint &cp)
{
  out<<cp.p1.x<<" "<<cp.p1.z<<" "<<cp.p2.x<<" "<<cp.p2.z<<endl;
  return out;
}

istream& top10::racing::operator>>(istream& in, CheckPoints &cps)
{
  int n;
  in>>n;
  CheckPoint tmp;

  cps.clear();
  for(;n>0; --n) {
    in>>tmp;
    cps.push_back(tmp);
  }

  return in;
}

ostream& top10::racing::operator<<(ostream& out, const CheckPoints &cps)
{
  out<<cps.size();

  for(CheckPoints::const_iterator p=cps.begin();
      p!=cps.end();
      ++p) {
    out<<(*p)<<endl;
  }

  return out;
}
