/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef WHEEL_H
#define WHEEL_H

#include "math/Curve.hh"
#include "math/Vertex.hh"
#include "math/Matrix.hh"

#include "util/Dumpable.hh"
#include "util/Debug.hh"
#include "SurfacePatch.hh"
#include <cassert>

namespace top10 {
  // Needed for debugging
  namespace ui_interactive {
    class Kart;
  };

  namespace track {
    class Track;
  };

  namespace physX {

    using top10::math::Curve2D;
    using top10::math::Vector;

    /*!
       A wheel with a tyre.
    */
    class Wheel: public top10::util::Dumpable, public top10::util::Debug
    {
    public:
      enum CollisionType { None, Contact, Collision };

      /*!
	 Default wheel: a go-kart wheel.
	 \param initial_orientation By default, up is (0,1,0), forward is (1,0,0), right is (0,0,1).
	        Use this parameter for other settings.
	 \param track pointer to the track
	 \param long_k longitudinal grip coefficient
	 \param side_k transversal grip coefficient
	 \param radius outer radius of the wheel
      */
      Wheel(top10::track::Track* track,
	    double long_k, double side_k,
	    double radius,
	    const char* name="Unnamed wheel",
	    bool debug=false);

      inline void setInitialOrientation(const top10::math::OrthoNorm3& M)
      {
	initial_orientation = M;
      }

      inline void setTransform(const top10::math::OrthoNorm3& M, Vector t) {
	orientation = M*initial_orientation; origin=t;
      }

      inline void setOrigin(Vector t) { origin = t; }

      inline void setHeight(double h) { height = h; }
      inline double getHeight() const {return height; }

      //! Handle collision with the ground
      /*!
	\param speed speed of the wheel
	\note Must be called before update()
      */
      void collideTrack(Vector speed, double dt);

      inline void clearForces() { external_force = Vector(0,0,0); }

      //! Apply external forces, but not from the ground
      inline void applyForce(Vector force) {external_force += force;}

      //! Indicates whether the wheel and the track are in contact
      inline CollisionType getGroundCollision() const { return ground_collision; }

      //! Update forces applied to the wheel from the ground, engine and brakes
      void computeForces(Vector speed, double dt);

      //! Return the reaction computed in computeForces
      inline Vector getForce() const { return global_force; }

      //! Return the torque applied on the wheel, as computed in computeForces.
      inline double getTorque() const { return torque; }

      //TODO...
      Vector getMomentum() const;

      //! During a collision, return the variation of speed
      inline Vector getImpulse() const { return impulse_speed; }

      /**
	 Get an average friction percentage.
	 Can be used to produce the sound's frequency of the tyre slipping on
	 concrete
      */
      double getFriction() const;

      //! Get the load on the wheel, can be used for the volume of the skidding sound
      double getLoad() const;

      /// Set orientation of the wheel (rad)
      void setAngle(double);
      /// Get orientation of the wheel (rad)
      double getAngle() const;

      /// Set torque applied by engine and brakes
      void setDriveTorque(double);
      /// Get torque applied by engine and brakes
      double getDriveTorque() const;

      /// Get rotational speed in rad/s
      double getAngularSpeed() const;
      inline void setAngularSpeed(double w) {w_speed = w;}

      Vector getFrictionForce() const;

      /// Get the bump factor of the surface under the wheel
      double getBumpFactor() const;
           
      void dumpTo(std::ostream&) const;
      void loadFrom(std::istream&);

    private:
      //! Forbidden operator
      Wheel& operator=(const Wheel& other) { assert(0); return *(Wheel*)0; }

      double gripFactor(const Vector& sliding_speed, const Vector& speed) const;

      /* Characteristics of a wheel */
      top10::math::OrthoNorm3 initial_orientation;

      /// The ground
      top10::track::Track* const track;

      /// Width of wheel
      double width;

      /// Radius
      double radius;

      //! Collision speed threshold (must be <0)
      double collision_speed;

      /// Maximum longitudinal "Stickyness" of the wheel to the ground. 
      double long_k;

      /** Values between 0 and 1 modifying the longitudinal stickyness depending on
	  the sliding factor */
      Curve2D grip_curve;

      /// Maximum transversal stickyness
      double side_k;

      /** Values between 0 and 1 modifying the transversal stickyness depending on
	  the tan value of the slip angle
      */
      Curve2D slip_angle_k_curve;

    private:
      /*
       * State of a wheel (changes every step)
       */
      Vector origin;
      top10::math::OrthoNorm3 orientation;
      double height;

      /// orientation of the wheel
      double angle;

      //! Force applied from anything but the ground
      Vector external_force;

      //! Contact/collision with the ground.
      CollisionType ground_collision;

      //! Impulse to correct speed after a collision
      Vector impulse_speed;

      /// Torque applied by the brakes or the engine
      double drive_torque;

      /// Rotation speed
      double w_speed;

      /// Surface the wheel lies over
      SurfacePatch surface;

      /// Speed of a point of the wheel relative to the ground
      Vector sliding_speed;

      /// Horizontal component of the force from the ground onto the wheel
      Vector friction_force;

      //! Force from the ground
      Vector global_force;

      //! Torque from the ground
      double torque;

      //! Vector relative to the orientation of the wheel
      Vector s1, s2, normal, up;

      // For debugging purpose
      friend class top10::ui_interactive::Kart;
    };
  };
};

#endif
