/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <assert.h>

#include "RigidBody.hh"

using namespace top10::physX;
using namespace std;

RigidBody::RigidBody(const char* name, bool debug):
  top10::util::Debug(name, debug),
  matrix_count(0)
{
}

Vector RigidBody::getLocalFromGlobal(Vector p) const
{
  return orient_inv*(p-getMassCenter()) + g_center;
}

Vector RigidBody::getGlobalFromLocal(Vector p) const
{
  return orient*(p-g_center) + getMassCenter();
}

void RigidBody::setMasses(const std::vector<Point>& points)
{
  g_center = Vector(0,0,0);
  mass = 0.0;

  // Compute mass center of the body
  for (vector<Point>::const_iterator p = points.begin();
       p != points.end();
       ++p) {
    g_center += p->mass * p->pos;
    mass += p->mass;
  }
  g_center /= mass;
  translation.pos = Vector(0,0,0);
  translation.mass = mass;

  // Compute matrix of inertia
  double Ixx=0, Iyy=0, Izz=0, Ixy=0, Ixz=0, Iyz=0;
  for (vector<Point>::const_iterator p = points.begin();
       p != points.end();
       ++p) {
    double x = p->pos.x - g_center.x;
    double y = p->pos.y - g_center.y;
    double z = p->pos.z - g_center.z;

    Ixx += p->mass*(y*y + z*z);
    Iyy += p->mass*(x*x + z*z);
    Izz += p->mass*(x*x + y*y);
    Ixy += p->mass*(x*y);
    Ixz += p->mass*(x*z);
    Iyz += p->mass*(y*z);
  }
  
  double vals[3][3] = { 
    { Ixx, -Ixy, -Ixz },
    {-Ixy,  Iyy, -Iyz },
    {-Ixz, -Iyz,  Izz }
  };
  inertia = top10::math::Matrix3(vals);
  inertia_inv = top10::math::inverse(inertia);
}

void RigidBody::clearForces()
{
  central_force = central_torque = Vector(0,0,0);
}

void RigidBody::applyForceAtL(Vector position, Vector force)
{
  central_force += force;
  Vector f2 = orient_inv*force;
#if 0
  if (fabs(f2.size()-force.size()) > 0.1) {
    std::cerr<<"f2: "<<f2.size()<<" != f: "<<force.size()<<std::endl;
    abort();
  }
#endif
  Vector T = f2^(g_center-position);
  central_torque += T;
}

void RigidBody::applyForceAt(Vector position, Vector force)
{
  applyForceAtL(getLocalFromGlobal(position), force);
}

void RigidBody::integrate(double dt)
{
  if (is_debug()) printDebugStart();

  // Update position
  translation.accel = central_force/mass;
  translation.integrate(dt);
  translation.shift();
  
  if (is_debug()) debug_stream<<"translation = "<<translation.pos<<" at speed "<<translation.speed<<std::endl;

  //  Update local base
#if 0
  matrix_count++;
  if (matrix_count >= 10) { orient.correct(); matrix_count = 0; }
#endif

#if 1
  Vector ws = orient*w_speed;
  double Nd[3][3] = {
    {0.0, -ws.z, ws.y},
    {ws.z, 0.0, -ws.x},
    {-ws.y, ws.x, 0.0}
  };
  top10::math::Matrix3 N(Nd);

  Vector dw = inertia_inv * (central_torque - (w_speed^(inertia*w_speed))) * dt; 
  Vector w_speed2 = w_speed + dw;
#if 0
  double Md[3][3] = {
    {0.0, -w_speed2.z, w_speed2.y},
    {w_speed2.z, 0.0, -w_speed2.x},
    {-w_speed2.y, w_speed2.x, 0.0}
  };
  top10::math::Matrix3 M(Md);
  orient=orient+dt*0.5*(N+M)*orient;
#else
  orient=orient+dt*N*orient;
#endif
  orient.correct();
  orient_inv = orient.transpose();

  // Update angular velocity
  if (is_debug()) debug_stream<<"d(w_speed) = "<<dw<<std::endl;
  w_speed = w_speed2;
#endif
  if (is_debug()) debug_stream<<"F = "<<central_force<<"T = "<<central_torque<<std::endl<<" ws = "<<w_speed<<std::endl;

  if (is_debug()) printDebugEnd();
}

Vector RigidBody::getSpeedAt(Vector pos) const
{
  return getSpeedAtL(getLocalFromGlobal(pos));
}

Vector RigidBody::getSpeedAtL(Vector pos) const
{
  return translation.speed + orient*((g_center-pos)^w_speed);
}

void RigidBody::dumpTo(ostream& str) const
{
  assert(0); // Not yet implemented
}

void RigidBody::loadFrom(istream& str)
{
  assert(0); // Not yet implemented
}

void RigidBody::applyLinearImpulseL(Vector impulse)
{
  translation.speed += orient*impulse / mass;
}

void RigidBody::applyImpulseAtL(Vector pos, Vector impulse)
{
  translation.speed += orient*impulse / mass;
  Vector r = pos - g_center;
  w_speed += (r^impulse)*inertia_inv;
}
