/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_MATH_RING_HH
#define TOP10_MATH_RING_HH

#include <cmath>
#include <cassert>

#include "util/error.hh"

namespace top10 {
  namespace math {

    /*!
      ANSI C++ does not allow floating point template parameters.
      Ok, let's use regular classes, then. This is the class of 0.
      Designed to be used to create the Angle type. Angle should have
      been declared as
      typedef Ring<0, 2*M_PI>. As it's not possible, it's declared as:
      typedef Ring<WeirdZERO, Weird2PI>

      Yes, it's the hell of a dirty hack, but it's lovely, isn't it ?
      (Ok, I admit it, I wrote this to scare Java people away)

      \sa Ring
    */
    class WeirdZERO {
    public:
      WeirdZERO() {};

      operator double() const { return 0.0; }
    };

    /*!
      Have a look at WeirdZERO. If you can't get what WeirdZERO is supposed
      to do, there is litlle chance you will succeed here.

      \sa WeirdZERO
    */
    class Weird2PI {
    public:
      Weird2PI() {};

      operator double() const { return 2*M_PI; }
    };

    /*!
      Represents a value inside [start, end[
      Useful for handling angles, for example
    */
    /* "ANSI C++ forbids floating-point template arguments" Why ??? :( */
    template <typename start, typename end>
    class Ring {
    public:
      DECL_ERROR(BadRange);

    public:
      Ring()
      {
	value = start();
      }

      Ring(double x)
      {
	if (double(end()) - double(start()) < 0) throw BadRange();

	value = x;
	
	fix();
      }

      Ring<start, end> complement() const {
	return Ring<start, end>(end() - double(value - start()));
      }

      Ring<start, end>&
      operator-=(Ring<start, end> other)
      {
	value -= other.value;
	fix();
	return *this;
      }

      Ring<start, end>&
      operator+=(Ring<start, end> other)
      {
	value += other.value;
	fix();
	return *this;
      }

      Ring<start, end>
      operator-(Ring<start, end> other) const
      {
	return Ring<start, end>(*this) -= other;
      }

      Ring<start, end>
      operator+(Ring<start, end> other) const
      {
	return Ring<start, end>(*this) += other;
      }

      Ring<start, end>&
      operator=(Ring<start, end> other)
      {
	value = other.value;
	return *this;
      }

      Ring<start, end>&
      operator=(double other)
      {
	value = other; fix();
	return *this;
      }
     
      operator double() const
      {
	return value;
      }

    private:
      double value;

      void fix()
      {
	double d = end()-start();
	while (value < start()) value += d;
	while (value >= end()) value -= d;
	assert(value >= start() && value < end());
      }

    };

    typedef Ring<WeirdZERO, Weird2PI> Angle;
  };

};

#endif
