#include <fstream>

#include "PolygonSet.hh"
#include "LwRead.hh"
#include "3dsRead.hh"
#include "util/error.hh"
#include "util/PathFinder.hh"
#include "math/Triangle.hh"

using top10::helpers::Polygon;
using top10::helpers::PolygonSet;
using top10::helpers::Substring_SurfaceFilter;
using top10::math::Vector;

Substring_SurfaceFilter::Substring_SurfaceFilter(std::string s, top10::physX::Surface p):
  substring(s), phys_surface(p)
{
}

bool Substring_SurfaceFilter::match(top10::helpers::Surface surf, top10::physX::Surface* p_out)
{
  if (surf.name.find(substring) != std::string::npos) {
    *p_out = phys_surface;
    return true;
  }
  return false;
}

void Substring_SurfaceFilter::writeOut(std::ostream& out) const
{
  out<<"SUBSTR \""<<substring<<"\" "<<phys_surface.grip<<" "<<phys_surface.drag<<" "<<phys_surface.bumps<<std::endl;
}

void Polygon::rotate(const top10::math::Matrix3& M)
{
  for (std::vector<Vector>::iterator v = p.begin();
       v != p.end();
       ++v)
    *v = M*(*v);
}

void Polygon::translate(top10::math::Vector t)
{
  for (std::vector<Vector>::iterator v = p.begin();
       v != p.end();
       ++v)
    *v = (*v) + t;
}

void PolygonSet::transform(top10::math::Matrix4 M)
{
  for (iterator poly = begin();
       poly != end();
       ++poly) {
    for (std::vector<Vector>::iterator v = poly->p.begin();
	 v != poly->p.end();
	 ++v) {
      *v = M * (*v);
    }
  }
}

void PolygonSet::setPhysSurfaces(SurfaceFilter& filter)
{
  top10::physX::Surface phys_surface;
  for (std::vector<Surface>::iterator surf = surfaces.begin();
       surf != surfaces.end();
       ++surf) {
    if (filter.match(*surf, &phys_surface)) surf->phys_surface = phys_surface;
  }
}

static bool isOfExtension(std::string filename, std::string ext)
{
  int s = filename.size();
  int s2 = ext.size();

  if (s <= s2) return false;

  return filename.substr(s-s2, s2) == ext;
}

PolygonSet* PolygonSet::loadNew(std::string filename)
{
  std::string path = top10::util::PathFinder::find(filename);
  if (path.empty()) throw top10::util::Error("Could not find ")+filename;

  if (isOfExtension(filename, "3ds")) {
    return new Read3DS(path.c_str());
  }
  else if (isOfExtension(filename, "lwo")) {
    std::ifstream ifs(path.c_str());
    return new LWRead(ifs);
  }
  throw top10::util::Error("Unknown file type [") + filename + "]";
  return 0;
}

int PolygonSet::intersectRay(Vector A, Vector V) const
{
  int res(-1);
  double dist(HUGE_VAL);

  int mesh_idx=0;
  for (std::vector<Mesh>::const_iterator mesh = meshes.begin();
       mesh != meshes.end();
       ++mesh, ++mesh_idx) {

    if (mesh->hidden) continue;

    for (int poly = mesh->begin; poly != mesh->end; ++poly) {
      const Polygon& polygon(at(poly));
      for (int i = 2; i<polygon.p.size(); ++i) {
	Vector dummy;
	double d;
	if (top10::math::Triangle(polygon.p[0], polygon.p[i-1], polygon.p[i]).intersectRay(A, V, dummy, d)) {
	  if (d < dist) {
	    dist = d;
	    res = mesh_idx;
	  }
	}
      }
    }
  }

  return res;
}

int PolygonSet::intersectRayPolygon(Vector A, Vector V) const
{
  int res(-1);
  double dist(HUGE_VAL);

  int mesh_idx=0;
  for (std::vector<Mesh>::const_iterator mesh = meshes.begin();
       mesh != meshes.end();
       ++mesh, ++mesh_idx) {

    if (mesh->hidden) continue;

    for (int poly = mesh->begin; poly != mesh->end; ++poly) {
      const Polygon& polygon(at(poly));
      for (int i = 2; i<polygon.p.size(); ++i) {
	Vector dummy;
	double d;
	if (top10::math::Triangle(polygon.p[0], polygon.p[i-1], polygon.p[i]).intersectRay(A, V, dummy, d)) {
	  if (d < dist) {
	    dist = d;
	    res = poly;
	  }
	}
      }
    }
  }

  return res;
}

int PolygonSet::findMeshIndex(std::string name) const
{
  int idx=0;
  for (std::vector<Mesh>::const_iterator it = meshes.begin();
       it != meshes.end();
       ++it, ++idx) {
    if (it->name == name) return idx;
  }
  return -1;
}
