/*
  Top 10, a racing simulator
  Copyright (C) 2000, 2001  Deneux Johann
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Deneux Johann:       deneux@ifrance.com
*/

#include <cstdlib>
#include <cstring>

#include "LwRead.hh"

using namespace top10::helpers;
using namespace std;

inline static string getString(std::istream& in)
{
  string str;
  char c;

  do {
    in.get(c);
    if (c) str+=c;
  } while (c && !in.eof());
  return str;
}

LWRead::LWRead(std::istream& _in):
  PolygonSet(),
  in(_in)
{
#if 0
#define CHECK_SIZE(a, size) if (sizeof(a) != size) cerr<<#a<<"is not "<<size<<endl
  // Check if types are the size we expect
  CHECK_SIZE(ID4, 4);
  CHECK_SIZE(I1, 1);
  CHECK_SIZE(I2, 2);
  CHECK_SIZE(I4, 4);
  CHECK_SIZE(U1, 1);
  CHECK_SIZE(U2, 2);
  CHECK_SIZE(U4, 4);
  CHECK_SIZE(F4, 4);
  CHECK_SIZE(VEC12, 12);
#undef CHECK_SIZE
#endif

  try {
    // I want exceptions to be thrown when something bad happens
    // Note: GNU libstdc++ does not seem to support this, yet.
    in.exceptions(ios::eofbit | ios::badbit | ios::failbit);
    
    ID4 form = getNextTag();
    if (!equals(form,"FORM")) throw ParseError("not a IFF LWO file");
    U4 size;
    in.read((char*)&size, 4);
    ftoh(size);

    U4 lwob;
    in.read((char*)&lwob, 4); size-=4;
    if (!equals(lwob,"LWOB")) throw ParseError("not a LWO file");

    int start_pos = in.tellg();
    U4 orig_size = size;
    for (;size>0;size-=4 /* getNextTag() ! */) {
      if (in.eof()) break;   // Stop loop if end of file reached earlier than expected
      if (!in.good()) throw ParseError("stream is bad");

      ID4 tag;
      tag=getNextTag();

      if (equals(tag,"PNTS")) size-=readVertices();
      else if (equals(tag, "SRFS")) size-=readSurfaceNames();
      else if (equals(tag, "POLS")) size-=readPolygons();
      else if (equals(tag, "CRVS")) size-=ignoreChunk("Curves not handled");
      else if (equals(tag, "PCHS")) size-=ignoreChunk("Patches not handled");
      else if (equals(tag, "SURF")) {
      	int c =readSurface(); size-=c;
      }
      else {
      	char tmp[5];
      	memcpy(tmp, (char*)&tag, 4);
      	tmp[4]=0;
      	// It seems that some files have garbage at the end. Stop it
      	if (tmp[0]<'A' || tmp[0]>'Z') {
      	  cerr<<"Warning, garbage tag found"<<endl;
      	  size=0;
      	}
      	else {
      	  size-=ignoreChunk("Unknown chunk");
      	}
      }

    }
    //    if (size!=0) throw LoadError(string("Bad file length"));
    if (size!=0) cout<<"Warning, EOF reached early ("<<size<<")"<<endl;

    fixateSurfaces();

    // Add the only mesh object
    Mesh new_mesh;
    new_mesh.begin = 0;
    new_mesh.end = this->size();
    new_mesh.name = "unnamed LW object";
    new_mesh.hidden = false;
    meshes.push_back(new_mesh);
  }
  catch (Error&) {throw;}
  //  catch (ios::eof) {throw LoadError("Unexpected end of file");}
  catch (...) {throw Error("Unknown stream exception");}

}

int LWRead::readPolygons()
{
  U4 size, ret;
  in.read((char*)&size, 4);
  ftoh(size);
  ret = size;

  while (size > 0) {
    if (!in.good()) throw ParseError("stream is bad");

    // Number of vertices
    U2 n_vert;
    in.read((char*)&n_vert, 2); size-=2;
    ftoh(n_vert);

    // Vertex indices
    U2 v0;

    Polygon poly;
    for (int i=0; i<n_vert; i++) {

      if (!in.good()) throw ParseError("stream is bad");

      in.read((char*)&v0, 2); ftoh(v0); size-=2;
      poly.p.push_back(Vector(vertices[v0].X, vertices[v0].Y, vertices[v0].Z));
    }

    // Face's color
    U2 surf_idx;
    in.read((char*)&surf_idx, 2); ftoh(surf_idx); size-=2;

    poly.surface_name = surf_idx -1;

    push_back(poly);
  }

  return ret+4;
}

LWRead::ID4 LWRead::getNextTag()
{
  if (!in.good()) throw ParseError("stream is bad");

  ID4 ret;
  in.read((char*)&ret, 4);

  return ret;
}

int LWRead::ignoreChunk(const char* reason)
{
  if (!in.good()) throw ParseError("stream is bad");

  U4 size;
  in.read((char*)&size, 4);
  ftoh(size);

  in.ignore(size);

  if (size%2) in.ignore(1);
  return (size%2)?(size +5):(size+4);
}

int LWRead::readSurface()
{
  if (!in.good()) throw ParseError("stream is bad");

  // Number of bytes still to read in the chunk
  U4 n_to_read, size;
  in.read((char*)&n_to_read, 4);
  ftoh(n_to_read);
  size=n_to_read;

  // Name of surface
  string name = getString(in);
  n_to_read-=name.size()+1;

  // If odd length, read extra null byte
  if ((name.size()+1) % 2 == 1) {
    char dummy;
    in.read(&dummy, 1);
    n_to_read--;
  }

  // Read sub-chunks
  do {
    if (!in.good()) throw ParseError("stream is bad");
    if (n_to_read < 4)
      throw ParseError("Bad remaining amount of surface bytes to read");

    n_to_read-=4;   // Tag size
    ID4 tag;
    tag=getNextTag();

    if (equals(tag, "COLR")) {
    	U2 size;
    	in.read((char*)&size, 2); n_to_read -= 2;
    	ftoh(size);
    	if (size != 4) throw ParseError(string("Bad COLR sub-chunk size"));
	
        COL4 color;
       	in.read((char*)&color, 4); n_to_read -= 4;
       	lw_surfaces[name] = color;
    }
    else {
    	char tmp[5];
    	memcpy(tmp, (char*)&tag, 4);
    	tmp[4] = 0;

    	// Ignore the sub chunk
    	U2 size;
    	in.read((char*)&size, 2); n_to_read -= 2;
    	ftoh(size);
	
    	in.ignore(size); n_to_read -= size;

    	if (size%2) {
    	  in.ignore(1);
    	  n_to_read-=1;
    	}
      }

  } while (n_to_read != 0);

  return size+4;
}

int LWRead::readSurfaceNames()
{
  if (!in.good()) throw ParseError("stream is bad");

  // Number of names
  U4 n_names;
  in.read((char*)&n_names, 4);
  ftoh(n_names);

  // Read names
  int n_read;
  for (int i=0; i<n_names; i+=n_read) {
    if (!in.good()) throw ParseError("stream is bad");

    string name = getString(in);

    // If odd length, read extra null byte
    n_read=name.size() +1;
    if (n_read % 2 == 1) {
      char dummy;
      in.read(&dummy, 1);
      n_read++;
    }

    // Add name
    surf_names.push_back(name);
  }

  return n_names+4;
}

int LWRead::readVertices()
{
  if (!in.good()) throw ParseError("stream is bad");

  // Get number of vertices
  U4 size;
  in.read((char*)&size, 4);
  ftoh(size);

  // Read vertices

  for (int i_vertex=0; i_vertex < size; i_vertex+=12) {
    if (!in.good()) throw ParseError("stream is bad");

    VEC12 vertex;
    in.read((char*)&vertex, 12);
    ftoh(vertex.X);
    ftoh(vertex.Y);
    ftoh(vertex.Z);

    // Add this vertex to the vertices
    vertices.push_back(vertex);
  }

  return size+4;
}

void LWRead::fixateSurfaces()
{
  for (std::vector<std::string>::const_iterator name = surf_names.begin();
       name != surf_names.end();
       ++name) {
    Surface new_surface;
    COL4 lw_surface = lw_surfaces[*name];
    new_surface.name = *name;
    new_surface.phys_surface = top10::physX::SurfaceSlip();
    new_surface.r = lw_surface.red;
    new_surface.g = lw_surface.green;
    new_surface.b = lw_surface.blue;
    new_surface.a = 255;
    new_surface.texture_filename.clear();
    surfaces.push_back(new_surface);
  }
}
