#!/usr/bin/env python
#
# This file is part of Tidydoc.
# Copyright (C) 2005, 2008, 2009 Nicolas Burrus
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from getopt import getopt
from glob import glob
import os, re, sys

import ac_config
import td_config
from td_config import config
import td_doctree
import td_dsc_to_xml
import td_html
import td_bibtex
import td_archive
from td_tools import *

def print_help():
  sys.stderr.write(
"""Usage: td-generate [options]

  Options

  -c, --config configfile  Config file to use.
  -v, --version            Print version number.
  --html                   Generate HTML pages only.
  --bibtex                 Generate Bibtex files only.
  --archive                Generate category archive.

Nicolas Burrus <nicolas.burrus@ensta.fr>
""")

def parse_command_line(argv):
  try: (options, args) =  getopt(argv[1:], "vc:",
                                 ["version", "config=", "html", "bibtex", "archive="])
  except:
    print_help()
    sys.exit(1)

  if len(args) != 0:
    print_help()
    sys.exit(1)

  cmd_options = { 'config_file' : None, 'html' : 0, 'bibtex' : 0, 'archive' : None }

  for option in options:
    if option[0] == "-v" or option[0] == "--version":
      print "Tidydoc " + ac_config.version
      print "Nicolas Burrus <nicolas.burrus@ensta.fr>"
      sys.exit(0)
    if option[0] == "-c" or option[0] == "--config":
      cmd_options['config_file'] = option[1]
    if option[0] == "--html":
      cmd_options['html'] = 1
    if option[0] == "--bibtex":
      cmd_options['bibtex'] = 1
    if option[0] == "--archive":
      cmd_options['archive'] = option[1]

  return cmd_options

def dump_categories(f, doctree, indent = 0):
  f.write("% " + " " * indent + rel_path(doctree.path) + "\n")
  for subdir in doctree.subdirs:
    dump_categories(f, subdir, indent + 2)

def generate_doctree():
  doctree = td_doctree.build(config['input_path'])
  td_doctree.dump(doctree)
  f = open(config['categories_path'], "w")
  dump_categories(f, doctree)
  f.close()
  return doctree  

def __main__():
  cmd_options = parse_command_line(sys.argv)
  try:
    td_config.read_config(cmd_options['config_file'])
  except:
    sys.stderr.write(str(sys.exc_info()[0]) + "\n")
    sys.stderr.write(str(sys.exc_info()[1]) + "\n")
    sys.exit(1)

  # If nothing was selected, select all
  if (cmd_options['html'] + cmd_options['bibtex'] == 0) and \
        not cmd_options['archive']:
    cmd_options['html'] = 1
    cmd_options['bibtex'] = 1

  # Convert dsc files.
  td_dsc_to_xml.run()

  doctree = generate_doctree()
  if cmd_options['html']:
    td_html.generate(doctree)
  if cmd_options['bibtex']:
    td_bibtex.generate(doctree)
  if cmd_options['archive']:
    td_archive.generate_category(doctree, 
                                 cmd_options['archive'], 
                                 cmd_options['archive'].replace('/','_') + ".zip")

if __name__ == "__main__":
  __main__()
  
