#!/usr/bin/env python
# This file is part of Tidydoc.
# Copyright (C) 2005, 2008, 2009 Nicolas Burrus
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from getopt import getopt
import os, sys

from td_config import config, read_config
import td_debug
import td_dsc
import td_tools

def print_help():
  sys.stderr.write(
"""Usage: td-upload [options] document_file description_file

  Options

  -c, --config configfile  Config file to use.
  -v, --version            Print version number.

Nicolas Burrus <nicolas.burrus@ensta.fr>
""")

def parse_command_line(argv):
  try: (options, args) =  getopt(argv[1:], "vc:", ["version", "config"])
  except:
    print_help()
    sys.exit(1)

  if len(args) != 2:
    print_help()
    sys.exit(1)

  cmd_options = {}
  cmd_options['doc_file'] = args[0]
  cmd_options['dsc_file'] = args[1]
  cmd_options['config_file'] = None

  for option in options:
    if option[0] == "-v" or option[0] == "--version":
      print "Tidydoc " + ac_config.version
      print "Nicolas Burrus <nicolas.burrus@ensta.fr>"
      sys.exit(0)
    if option[0] == "-c" or option[0] == "--config":
      cmd_options['config_file'] = option[1]

  return cmd_options

def get_upload_cmd(orig_doc_path, orig_dsc_path):
  dsc_sections = td_dsc.parse(orig_dsc_path)

  # Compute destination filenames.
  final_doc_name = td_dsc.docname_from_dsc(dsc_sections, orig_doc_path)
  final_dsc_name = final_doc_name + ".dsc"

  # Extract the first category.
  categories = dsc_sections['Categories']
  if len(categories) == 0:
    categories = ["unsorted"]

  first_category = categories[0].rstrip("\r\n")
  first_destdir = os.path.join(config['upload_dest_path'], first_category)

  # Add files into the directories corresponding to the categories.
  upload_cmd = ""
  for category in categories:
    destdir = os.path.join(config['upload_dest_path'], category)

    # For the first category, the file will be copied.
    if (category == first_category):
      upload_cmd += config['upload_commands'] % locals()
    else:
      # For the other categories, a link is enough.
      rel_first_destdir = ".."
      for i in range(0, category.count('/')):
        rel_first_destdir += "/.."
      rel_first_destdir += '/' + first_category
      upload_cmd += config['link_commands'] % locals()

  log = "`" + orig_doc_path + "' ==> `" + final_doc_name + "'"
  for category in categories:
    log = log + " " * len(orig_doc_path) + "   ==> " + category + "/"    
  return (upload_cmd, log)

def upload(orig_doc_path, orig_dsc_path, ask_for_confirmation = None):
  (cmd, log) = get_upload_cmd(orig_doc_path, orig_dsc_path)

  if (ask_for_confirmation):
    print log
    print "Is this ok ? [y/n]"
    ans = sys.stdin.readline()
    if (ans.lower() != "y\n"):
      print "Aborting..."
      sys.exit (1)
    else:
      print "Ok, copying files..."

  td_debug.debug(log)

  # Execute commands
  (status, lines) = td_tools.exec_commands(cmd)
  if status != 0:
    raise "Error while uploading: " + "\n".join(lines)

def __main__():
  cmd_options = parse_command_line(sys.argv)
  read_config(cmd_options['config_file'])
  upload(cmd_options['doc_file'], cmd_options['dsc_file'])

if __name__ == "__main__":
  __main__()
