# This file is part of Tidydoc.
# Copyright (C) 2005, 2008, 2009 Nicolas Burrus
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from glob import glob
import subprocess
import os.path
import re
import os
import stat

from td_config import config

def listdirs(directory):
  entries = os.listdir(directory)
  subdirs = []
  for entry in entries:
    if os.path.isdir(os.path.join(directory, entry)):
      subdirs.append(entry)
  return subdirs

def replace_in_template(tpl_file, subs):
  """Replace entries in tpl_file using subs"""
  f = open(tpl_file, "r")
  s = f.read()
  f.close()
  for key,item in subs.items():
    s = s.replace(key, item) 
  return s

def rel_path(input_path):
  dir_basename = re.sub(config['input_path'], '', input_path)
  if len(dir_basename) > 0 and dir_basename[0] == '/':
    dir_basename = dir_basename[1:]
  return dir_basename

def get_output_path(input_path):
  return os.path.join(config['output_path'], rel_path(input_path))

def get_site_path(input_path):
  return os.path.join(config['site_root'], rel_path(input_path))

def exec_interactive_commands(cmds):
  status = subprocess.call(cmds, shell=True)
  if status != 0:
    msg = "Could not run `%s'." % cmds
    raise Exception(msg)
  return status

def exec_commands(cmds):
  p = subprocess.Popen(cmds, shell=True,
                       stdout=subprocess.PIPE, 
                       stderr=subprocess.STDOUT, 
                       close_fds=True)
  if not p:
    raise Exception("Could not open Pipe!")
  child_stdout_and_stderr = p.stdout
  lines = child_stdout_and_stderr.readlines()
  status = p.wait()
  if status != 0:
    msg = "Could not run `%s': %s" % (cmds,"\n".join(lines))
    raise Exception(msg)
  return status, lines

def guess_desc_from_pdf(desc, doc_path):
  cmd = "pdfinfo \"%s\" 2> /dev/null | grep Title: | sed 's/Title:[ \t]\+//'\n" \
        % doc_path
  try:
    (status, output) = exec_commands(cmd)
    if status == 0 and len(output) > 0:
      desc['title'] = output[0].strip("\n")
  except:
    pass

  cmd = "pdfinfo \"%s\" 2> /dev/null | grep Author: | sed 's/Author:[ \t]\+//'\n" \
        % doc_path
  try:
    (status, output) = exec_commands(cmd)
    if status == 0 and len(output) > 0:
      desc['author'] = output[0].strip("\n")
  except:
    pass

  cmd = "pdfinfo \"%s\" 2> /dev/null | grep Pages: | sed 's/Pages:[ \t]\+//'\n" \
        % doc_path
  try:
    (status, output) = exec_commands(cmd)
    if status == 0 and len(output) > 0:
      desc['pages'] = output[0].strip("\n")
  except:
    pass

  cmd = "pdfinfo \"%s\" 2> /dev/null | grep CreationDate:" % doc_path
  try:
    (status, output) = exec_commands(cmd)
    if status == 0 and len(output) > 0:
      m = re.match("^.*(\d{4})$", output[0].rstrip())
      if m:
        desc['date'] = m.group(1)
  except:
    pass
  return desc

def guess_desc_from_doc(doc_path):
  desc = { 'author' : "",
           'date'   : "",
           'pages'  : "",
           'title'  : "",
           'conf'   : ""}
  if re.match(".*\.pdf$", doc_path):
    return guess_desc_from_pdf(desc, doc_path)
  return desc

def is_more_recent(f1, f2):
  time1 = os.stat(f1)[stat.ST_MTIME]
  time2 = os.stat(f2)[stat.ST_MTIME]
  return (time1 >= time2)
