#!/usr/bin/env python
#
# Copyright (C) 2005, 2006, 2008, 2009 Nicolas Burrus
# This file is part of Tidydoc.
#
# Tidydoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Tidydoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Foobar; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os, re, sys

from td_tools import *
import td_config
import td_xml
import td_doctree
import td_bibtex
import td_debug
from td_config import config
import ac_config

def generate_subdirs(doctree):
  subdirs_tpl = os.path.join(config['templates_path'], 'html/subdir.tpl')
  html_output = ""
  for subdir in doctree.subdirs:
    name = os.path.basename(subdir.path)
    subs = {
      '%%LINK%%'   : os.path.join(name, 'index.html'),
      '%%NAME%%'   : name,
      '%%NUMBER%%' : str(td_doctree.nb_docs(subdir)),
      }
    html_output += replace_in_template(subdirs_tpl, subs)
  return html_output

def generate_document_index(doctree):
  link_tpl = os.path.join(config['templates_path'], 'html/link.tpl')
  html_output = ""
  
  for doc in doctree.docs:
    subs = {
      '%%FIRST_AUTHOR%%' : doc.content['author'][0],
      '%%AUTHORS%%' : ", ".join(doc.content['author']),
      '%%LINK%%' : '#' + doc.content['title'][0],
      '%%NAME%%' : doc.content['title'][0]
      }
    html_output += replace_in_template(link_tpl, subs)
  return html_output

def shorten_string(m):
  s = m.group(2)
  if (len(s) > 48):
    s = s[0:23] + "..." + s[len(s)-1-23:len(s)]
  return m.group(1) + s + m.group(3)

def generate_html_code(doc, dirpath):
  document_tpl = os.path.join(config['templates_path'], 'html/document.tpl')
  short_filename = re.sub(r'(.*\.)(.*)(\.\w+)$', 
                          shorten_string,
                          doc.content['file'][0])
  subs = {
        '%%ANCHOR%%'   : doc.content['title'][0],
        '%%AUTHORS%%'  : "<br>".join(doc.content['author']),
        '%%FIRST_AUTHOR%%' : doc.content['author'][0],
        '%%CONF%%' : doc.content['conf'][0],
        '%%DOC_ID%%' : doc.content['file'][0],
        '%%DATE%%'     : doc.content['date'][0],
        '%%LANGUAGE%%' : doc.content['language'][0],
        '%%DOCPATH%%'  : dirpath + '/' + doc.content['file'][0],
        '%%LINK%%'     : config['site_root'] + '/' + dirpath + '/' + doc.content['file'][0],
        '%%LINK_NAME%%' : short_filename,
        '%%SUMMARY%%'  : re.sub("\n", "<br>\n", doc.content['summary'][0]),
        '%%COMMENT%%'  : doc.content['comment'][0],
        '%%URL%%'      : doc.content['url'][0],
        '%%TITLE%%'    : doc.content['title'][0],
        '%%NBPAGES%%'  : doc.content['nbpages'][0],
        '%%BIBTEX%%'   : re.sub("\n", "<br>\n",
                                td_bibtex.generate_bibtex_entry(doc)),
        }
  doc_html = replace_in_template(document_tpl, subs)
  return doc_html

def generate_document_list(doctree):
  html_output = ""

  dirpath = os.path.abspath(doctree.path)
  dirpath = dirpath.replace(os.path.abspath(config['input_path']), "").lstrip('/')
 
  current_id = 0
  for doc in doctree.docs:
    doc_html = ""
    cache_file = config['input_path'] + '/' + dirpath + '/' + doc.content['file'][0] + ".html"
    xml_file = config['input_path'] + '/' + dirpath + '/' + doc.content['file'][0] + ".xml"
    if os.path.exists(cache_file) \
       and os.path.exists(xml_file) \
       and is_more_recent(cache_file, xml_file):
      doc_html = open(cache_file, "r").read().decode("utf_8")
    else:  
      doc_html = generate_html_code(doc, dirpath)
      # Cache
      f = open(cache_file, "w")
      f.write(doc_html.encode("utf_8"))
      f.close()
    html_output += doc_html
  return html_output

def generate_html_file(doctree, fname):
  """Generate html file fname"""
  tpl = os.path.join(config['templates_path'], 'html/' + fname + '.tpl')
  input_path = config['input_path']
  output_path = config['output_path']
  
  subdirs_html = generate_subdirs(doctree)
  docs_list_html = generate_document_list(doctree)
  docs_index_html = generate_document_index(doctree)
  
  dir_basename = get_output_path(doctree.path)
  output_file = os.path.join(dir_basename, fname)
  subs = {
      '%%DOCS_LIST%%'     : docs_list_html,
      '%%DOCS_INDEX%%'    : docs_index_html,
      '%%SUBDIRS_INDEX%%' : subdirs_html,
      '%%SITE_ROOT%%'     : config['site_root'],
      '%%VERSION%%'       : ac_config.version
      }
  html_content = replace_in_template(tpl, subs)
  try:
    os.makedirs(os.path.dirname(output_file))
  except:
    pass

  f = open(output_file, "w")
  f.write(html_content.encode("utf_8"))
  f.close()

def generate_child_node(doctree):
  for f in config['directories_html_files']:
    generate_html_file(doctree, f)
  for subdir in doctree.subdirs:
    generate_child_node(subdir)

def generate_html(doctree):
  """Generate html files"""
  for f in config['site_root_html_files']:
    generate_html_file(doctree, f)
  generate_child_node(doctree)

# Treeview support
def generate_treeview_text(doctree, depth):
  link = os.path.join(get_site_path(doctree.path),
                      config['treeview_subdirs_link'])
  # Treeview need the first folder to be called foldersTree :(
  if depth == 1:
    root_depth = "foldersTree"
  else:
    root_depth = "depth" + str(depth)
  js = " " * (depth * 2) + 'depth%s = insFld(%s, gFld("%s", "%s"))\n' \
       % (str(depth+1), root_depth, os.path.basename(doctree.path), link)

  for subdir in doctree.subdirs:
    js += generate_treeview_text(subdir, depth+1)

  if config['treeview_multiframes']:
    link_mode = "R"
  else:
    link_mode = "S"

  for doc in doctree.docs:
    link = get_site_path(doctree.path) + "/" + config['treeview_subdirs_link']
    link += "#" + doc.content['title'][0]
    js += " " * (depth * 2) + 'insDoc(depth%s, gLnk("%s", "%s", "%s"))\n' \
          % (str(depth+1), link_mode, doc.content['title'][0], link)

  return js

def generate_treeview(doctree):
  treeview_path = os.path.join(config['output_path'], 'treeview')
  treeview_tpl = os.path.join(config['templates_path'], 'html/treeview.tpl')
  try:
    os.makedirs(treeview_path)
  except:
    pass

  root = os.path.join(config['site_root'], config['treeview_subdirs_link'])
  js = 'foldersTree = gFld("<b>Index</b>", "%s")\n' % root
  for subdir in doctree.subdirs:
    js += generate_treeview_text(subdir, 1)

  # FIXME: factorize with generate_treeview_text.
  # Treeview requires a special treatment for the first item :(
  for doc in doctree.docs:
    link = get_site_path(doctree.path) + "/" + config['treeview_subdirs_link']
    link += "#" + doc.content['title'][0]
    js += 'insDoc(foldersTree, gLnk("R", "%s", "%s"))\n' \
          % (doc.content['title'][0], link)
  
  f = open(os.path.join(treeview_path, 'docNodes.js'), "w")
  subs = { '%%TREE_CONTENT%%' : js }
  f.write(replace_in_template(treeview_tpl, subs).encode("utf_8"))
  f.close()

def generate(doctree):
  generate_html(doctree)
  generate_treeview(doctree)
