/* ***************************************************************************

    The Mneme Project
    http://theMnemeProject.org/  http://nonprofit-crm.org/
    Copyright (C) 2009 - 2014  Eric Chadbourne
    eric.chadbourne@gmail.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*************************************************************************** */

CREATE TABLE admin (
    admin_var character varying NOT NULL,
    admin_value character varying
);

COMMENT ON TABLE admin IS 'holds values for mneme settings';

COMMENT ON COLUMN admin.admin_var IS 'same name as the php variable';

COMMENT ON COLUMN admin.admin_value IS 'the value of the variable';

CREATE TABLE constituents_campaigns (
    pri_key integer NOT NULL,
    campaign_id character varying NOT NULL,
    campaign_descrip character varying
);

COMMENT ON TABLE constituents_campaigns IS 'to keep track of campaign id''s and descriptions';

CREATE TABLE constituents_campaigns_active (
    pri_key integer NOT NULL,
    campaign_id character varying,
    user_id integer
);

COMMENT ON TABLE constituents_campaigns_active IS 'for managing and segmenting campaigns';

CREATE SEQUENCE constituents_campaigns_active_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE constituents_campaigns_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_interactions (
    interaction_id integer NOT NULL,
    campaign_id character varying(50) NOT NULL,
    user_id integer NOT NULL,
    int_type character varying(30),
    medium character varying(30),
    int_date date,
    amount money,
    int_notes character varying(1000),
    amount_type character(2),
    amount_items character varying(30)
);

COMMENT ON COLUMN constituents_interactions.amount_type IS 'this is used to keep track of the type of donation.  such as hard or soft.  hard means actual money, soft could be the donation was made by another in this persons name.';

COMMENT ON COLUMN constituents_interactions.amount_items IS 'this is for the amount or number if items purchased';

CREATE SEQUENCE constituents_interactions_interaction_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_interactions_types (
    pri_key integer NOT NULL,
    type_medium character varying NOT NULL,
    type_name character varying NOT NULL,
    type_description character varying
);

COMMENT ON TABLE constituents_interactions_types IS 'for the list of interaction types and mediums on interaction page';

COMMENT ON COLUMN constituents_interactions_types.type_medium IS 'which drop down are we using, type or medium';

COMMENT ON COLUMN constituents_interactions_types.type_name IS 'name used in drop down';

COMMENT ON COLUMN constituents_interactions_types.type_description IS 'brief description of usage';

CREATE SEQUENCE constituents_interactions_types_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_members (
    user_id integer NOT NULL,
    contact_type character varying(20) NOT NULL,
    donotemail character varying(2),
    donotphone character varying(2),
    donotmail character varying(2),
    donotsms character varying(2),
    donottrade character varying(2),
    display_name character varying(100),
    nick_name character varying(50),
    preferredcommunicationmethod character varying(300),
    source_of_contact_data character varying(50),
    first_name character varying(50),
    middle_name character varying(50),
    last_name character varying(50),
    job_title character varying(70),
    birth_date date,
    is_deceased character varying(2),
    organization_name character varying(100),
    gender character varying(2),
    individual_prefix character varying(6),
    individual_suffix character varying(6),
    current_employer character varying(100),
    street_address character varying(100),
    supplemental_address_1 character varying(100),
    supplemental_address_2 character varying(100),
    city character varying(50),
    state1 character varying(50),
    country character varying(10),
    phone character varying(30),
    email character varying(50),
    website character varying(50),
    group_list_newsletter character varying(2),
    group_list_annual_appeal_mail character varying(2),
    zip_code character varying(10),
    group_donors character varying(2),
    group_staff character varying(2),
    group_volunteer character varying(2),
    alt_street_address character varying(100),
    alt_supplemental_address_1 character varying(100),
    alt_supplemental_address_2 character varying(100),
    alt_city character varying(50),
    alt_state1 character varying(50),
    alt_country character varying(10),
    alt_zip_code character varying(10),
    alt_phone character varying(30),
    alt_email character varying(50),
    alt_website character varying(50),
    alt_choice_address character varying(2),
    alt_choice_phone character varying(2),
    alt_choice_email character varying(2),
    alt_choice_website character varying(2),
    fax character varying(30),
    group_list_annual_appeal_email character varying(2)
);

COMMENT ON COLUMN constituents_members.preferredcommunicationmethod IS 'used for notes';

COMMENT ON COLUMN constituents_members.street_address IS 'this is for business addresses';

COMMENT ON COLUMN constituents_members.phone IS 'this is for business telephone numbers';

COMMENT ON COLUMN constituents_members.email IS 'this is for business email addresses';

COMMENT ON COLUMN constituents_members.website IS 'this is for business related websites';

COMMENT ON COLUMN constituents_members.alt_street_address IS 'this is for home addresses';

COMMENT ON COLUMN constituents_members.alt_phone IS 'this is for home telephone numbers';

COMMENT ON COLUMN constituents_members.alt_email IS 'this is for home or personal email addresses';

COMMENT ON COLUMN constituents_members.alt_website IS 'this is for home or personal websites';

COMMENT ON COLUMN constituents_members.alt_choice_address IS 'used to set business or home addresses';

COMMENT ON COLUMN constituents_members.alt_choice_phone IS 'used to set business or home phone';

COMMENT ON COLUMN constituents_members.alt_choice_email IS 'used to set business or home email';

COMMENT ON COLUMN constituents_members.alt_choice_website IS 'used to set business or home websites';

COMMENT ON COLUMN constituents_members.fax IS 'this is for business fax numbers';

CREATE TABLE constituents_members_fields (
    field_id integer NOT NULL,
    user_id integer NOT NULL,
    test character varying(100)
);

COMMENT ON TABLE constituents_members_fields IS 'this table is used to store custom user defined fields relating to constituents';

CREATE SEQUENCE constituents_members_fields_field_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_members_fnames (
    pri_key integer NOT NULL,
    column_name character varying(100) NOT NULL,
    human_name character varying(100) NOT NULL
);

COMMENT ON TABLE constituents_members_fnames IS 'this table supports table constituents_members_fields.  it is a human friendly string for the column name in constituents_members_fields.';

COMMENT ON COLUMN constituents_members_fnames.column_name IS 'this is the name of the column';

COMMENT ON COLUMN constituents_members_fnames.human_name IS 'this is the human readable name used on the web pages';

CREATE SEQUENCE constituents_members_fnames_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE constituents_members_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_members_fields_int (
    field_id integer NOT NULL,
    interaction_id integer NOT NULL,
    test character varying(100)
);

COMMENT ON TABLE constituents_members_fields_int IS 'this table is used to store custom user defined fields relating to interactions';

CREATE SEQUENCE constituents_members_fields_int_field_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE constituents_members_fnames_int (
    pri_key integer NOT NULL,
    column_name character varying(100) NOT NULL,
    human_name character varying(100) NOT NULL,
    group_name_int character varying(100) NOT NULL
);

COMMENT ON TABLE constituents_members_fnames_int IS 'this table supports table constituents_members_fields_int.  it is a human friendly string for the column name in constituents_members_fields_int.';

COMMENT ON COLUMN constituents_members_fnames_int.column_name IS 'this is the name of the column';

COMMENT ON COLUMN constituents_members_fnames_int.human_name IS 'this is the human readable name used on the web pages';

COMMENT ON COLUMN constituents_members_fnames_int.group_name_int IS 'used to select which group of custom fields to display.  staff can select custom fields.';

CREATE SEQUENCE constituents_members_fnames_int_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE groups (
    user_group_1 character varying(10) NOT NULL,
    user_group_2 character varying(10) NOT NULL,
    description_group_1 character varying(60) NOT NULL,
    description_group_2 character varying(60)
);

COMMENT ON TABLE groups IS 'here we can keep track of our various user groups.  all users will be in a group.';

CREATE TABLE simple_page (
    pri_key integer NOT NULL,
    page_name text,
    page_value text
);

COMMENT ON TABLE simple_page IS 'this table is for the module of the same name';

CREATE SEQUENCE simple_page_pri_key_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE users (
    first_name character varying(20) NOT NULL,
    last_name character varying(20) NOT NULL,
    middle_name character varying(20),
    user_group_1 character varying(10) NOT NULL,
    user_group_2 character varying(10),
    email character varying(40) NOT NULL,
    login_date timestamp without time zone,
    password character varying(200) NOT NULL,
    user_id integer NOT NULL
);

CREATE SEQUENCE users_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;
    
CREATE TABLE case_notes_users (
     email character varying(40) NOT NULL
);

COMMENT ON TABLE case_notes_users IS 'this simple table is to list what staff are allowed to access the case_notes module';

CREATE TABLE case_notes (
    note_id SERIAL PRIMARY KEY,
    user_id integer NOT NULL,
    int_date date,
    int_notes text
);

COMMENT ON COLUMN case_notes.int_notes IS 'private case notes for limited staff members';

ALTER TABLE ONLY constituents_campaigns ALTER COLUMN pri_key SET DEFAULT nextval('constituents_campaigns_pri_key_seq'::regclass);

ALTER TABLE ONLY constituents_campaigns_active ALTER COLUMN pri_key SET DEFAULT nextval('constituents_campaigns_active_pri_key_seq'::regclass);

ALTER TABLE ONLY constituents_interactions ALTER COLUMN interaction_id SET DEFAULT nextval('constituents_interactions_interaction_id_seq'::regclass);

ALTER TABLE ONLY constituents_interactions_types ALTER COLUMN pri_key SET DEFAULT nextval('constituents_interactions_types_pri_key_seq'::regclass);

ALTER TABLE ONLY constituents_members ALTER COLUMN user_id SET DEFAULT nextval('constituents_members_user_id_seq'::regclass);

ALTER TABLE ONLY constituents_members_fields ALTER COLUMN field_id SET DEFAULT nextval('constituents_members_fields_field_id_seq'::regclass);

ALTER TABLE ONLY constituents_members_fnames ALTER COLUMN pri_key SET DEFAULT nextval('constituents_members_fnames_pri_key_seq'::regclass);

ALTER TABLE ONLY constituents_members_fields_int ALTER COLUMN field_id SET DEFAULT nextval('constituents_members_fields_int_field_id_seq'::regclass);

ALTER TABLE ONLY constituents_members_fnames_int ALTER COLUMN pri_key SET DEFAULT nextval('constituents_members_fnames_int_pri_key_seq'::regclass);

ALTER TABLE ONLY simple_page ALTER COLUMN pri_key SET DEFAULT nextval('simple_page_pri_key_seq'::regclass);

ALTER TABLE ONLY users ALTER COLUMN user_id SET DEFAULT nextval('users_user_id_seq'::regclass);

ALTER TABLE ONLY admin
    ADD CONSTRAINT admin_pkey PRIMARY KEY (admin_var);

ALTER TABLE ONLY constituents_members
    ADD CONSTRAINT constit_pri_key PRIMARY KEY (user_id);

ALTER TABLE ONLY constituents_interactions
    ADD CONSTRAINT constituents_interactions_pkey PRIMARY KEY (interaction_id);

ALTER TABLE ONLY constituents_members_fields
    ADD CONSTRAINT constituents_members_fields_pri_key PRIMARY KEY (field_id);

ALTER TABLE ONLY constituents_members_fnames
    ADD CONSTRAINT constituents_members_fnames_pri_key PRIMARY KEY (pri_key);
    
ALTER TABLE ONLY constituents_members_fields_int
    ADD CONSTRAINT constituents_members_fields_int_pri_key PRIMARY KEY (field_id);

ALTER TABLE ONLY constituents_members_fnames_int
    ADD CONSTRAINT constituents_members_fnames_int_pri_key PRIMARY KEY (pri_key);

ALTER TABLE ONLY constituents_campaigns
    ADD CONSTRAINT pri_key_camp PRIMARY KEY (pri_key);

ALTER TABLE ONLY constituents_campaigns_active
    ADD CONSTRAINT pri_key_camp_act PRIMARY KEY (pri_key);

ALTER TABLE ONLY constituents_interactions_types
    ADD CONSTRAINT pri_key_int_type PRIMARY KEY (pri_key);

ALTER TABLE ONLY simple_page
    ADD CONSTRAINT simple_page_pkey PRIMARY KEY (pri_key);

CREATE INDEX fki_group ON users USING btree (user_group_1);

CREATE INDEX user_1_index ON groups USING btree (user_group_1);

ALTER TABLE ONLY constituents_interactions
    ADD CONSTRAINT constituents_interactions_user_id_fkey FOREIGN KEY (user_id) REFERENCES constituents_members(user_id);

--
-- Populate a couple of tables to get us running
--

insert into groups (user_group_1, user_group_2, description_group_1, description_group_2)
values 
('admin','admin','admin of website','admin of website'),
('exec','exec','executive staff','executive staff all access'),
('user','user_serv','regular user service','all access service'),
('user','user_dev','regular user dev','all access development'),
('volunteer','vol_dev','volunteer','limited development no reports'),
('volunteer','vol_serv','volunteer','limited service no reports');

insert into admin (admin_var, admin_value)
values 
('admin_about_us','<p>The mneme project is a is fast, light, and user-friendly constituent relationship management system built on PHP, PostgreSQL and Gnu/Linux.  <a href="http://theMnemeProject.org/">http://theMnemeProject.org/</a>
</p>'),
('admin_header_title','Mneme Database'),
('admin_header_meta_desc','a mneme project site'),
('admin_header_meta_keywords','gnu/linux, php, postgresql'),
('admin_email','you@yours.org');

insert into constituents_campaigns (campaign_id, campaign_descrip)
values 
('Emtpy ID','Used when a campaign ID is not needed'),
('2013 Test','Default install test campaign.  Can be deleted.');

insert into constituents_members_fnames
(column_name, human_name)
values
('test', 'An example field');

insert into constituents_interactions_types 
(type_medium, type_name, type_description)
values 
('type','Asked',''),
('type','Contacted',''),
('type','Donated','For general donations'),
('type','Other','general non specific type'),
('type','Paid',''),
('type','Pledged',''),
('type','RSVP',''),
('type','Sponsored',''),
('type','Thank You','For thank you mail'),
('type','Visit',''),
('medium','Cash',''),
('medium','Check',''),
('medium','Credit',''),
('medium','Email',''),
('medium','Letter',''),
('medium','Other','general non specific medium'),
('medium','PayPal',''),
('medium','Payroll',''),
('medium','Person',''),
('medium','Telephone',''),
('medium','Website','');

insert into simple_page 
(page_name, page_value)
values
(
'interactions',
'<span class="colorme">An interaction is any significant exchange with a constituent that is usually associated with a campaign.</span>
<br /><br />
As an example lets say we have a campaign called “2013 Annual Appeal”.  
<br /><br />
We added our groups to the campaign and created the email list in the settings section.  This automatically created an interaction where the campaign_id = “2013 Annual Appeal”, the int_type (interaction type) = ''Asked'', medium = ''Email'', and int_date (interaction date) = today.
<br /><br />
Our constituent, let''s call him Mr. Stallman, mails us a check for $100.  When the check arrives a staff member searches for Mr. Stallman on the search page.  This is a good time to check for duplicate entries or inaccurate information.  The staff member then creates an interaction, selecting our campaign and other relevant information.
<br /><br />
When we run a report on the 2013 Annual Appeal we will see two interactions for Mr. Stallman.  One we emailed him and asked for a donation.  Two we will see a donation for $100.  This can be used in deciding which future campaigns to include Mr. Stallman in.'
),

(
'help',
'Here is a list of the current help pages.  Please contact the IT department if you would like any changes.

<ul class="list">

<li><span>1</span><a href="./show.php?name=overview">Overview</a></li>

<li><span>2</span><a href="./show.php?name=constituents">Constituents</a></li>

<li><span>3</span><a href="./show.php?name=interactions">Interactions</a></li>

<li><span>4</span><a href="./show.php?name=groups">Groups</a></li>

<li><span>5</span><a href="./show.php?name=campaigns">Campaigns</a></li>

</ul>'
),

('overview',
'<p><b>A Constituent Relationship Management System</b></p>
<br />
Terminology:
<ul>
	<li>A <span class="colorme">constituent</span> is defined here
	has anybody you want to keep track of in the database.</li>
	<li>A <span class="colorme">group</span> is used to associate a
	constituent with something. Such as you can create a group called
	''donors''. If a constituent donates money you could add them to the
	donor group. A constituent can be a member of more than one group.</li>
	<li>An <span class="colorme">interaction</span> is any
	significant exchange with a constituent. Such as a constituent
	donates money. We could add them to the donor group and create an
	interaction stating when and for how much the donation was for.</li>
	<li>A <span class="colorme">campaign</span> is when you desire to
	interact with multiple constituents. An example of this could be an
	annual appeal. We could create a campaign called the ''2012 Annual
	Appeal''. We then mail all constituents in the group ''donors'' and
	create an interaction stating that we mailed them. If they reply
	that would be an interaction.</li>
</ul>

<p>The four fundamental terms above can be
used to analyze previous efforts and help refine future ones.</p>
<br />
Benefits:
<ul>
	<li><span class="colorme">Data retention</span> over multiple years and staff members.  By standardizing the 
	way we store information we minimize data loss.  No reinventing the wheel every couple 
	of years.</li>
	<li><span class="colorme">Collaborative</span> work between departments and staff members. 
	Multiple staff members can work on the same project without duplication of 
	effort.</li>
	<li>No <span class="colorme">vendor lock-in</span>.  By using free and open source software we can 
	shop around to multiple software companies instead of using a single 
	proprietary solution that only one company supports.</li>
	<li>This application <span class="colorme">costs</span> far less than competing solutions. No licensing or monthly fees.</li>
	<li>Creating custom <span class="colorme">reports</span> is easy for those with a basic understanding of PHP and SQL.</li>
	<li><span class="colorme">Data analysis</span> for marketing and development campaigns.  Measuring past campaigns 
	and segmenting constituents offer powerful tools for actionable information.</li>
	<li>Connecting with <span class="colorme">other databases</span> such as MySQL or Oracle is possible.</li>
	<li>This application is <span class="colorme">easy to maintain</span>.  It uses some of the most popular and long lasting 
	software available.  There is a large talent pool of technologists who know how to use 
	this software.</li>
</ul>'
),

('constituents',
'<p>A <span class="colorme">constituent</span> is defined here has anybody you want to keep track of in the database.  It could be a donor, volunteer, politician, staff member, a client, and more.</p>
<br />
<p>Here is an explanation of some of the more import data points.</p>
<br />
<p><span class="colorme">Contact type</span>.  Individual or Organization.  An example of an individual could be client receiving a service.  An organization could be a local business.</p>
<br />
<p><span class="colorme">Source of Contact Data</span> is a useful way to keep track of how the record was initially entered into the database.</p>
<br />
<p>The <span class="colorme">notes field</span> can be used for any general text.  Underneath the notes field there are several check boxes.  Such as if you have a constituent who has requested to never be called you could check to the <span class="colorme">"do not phone"</span> box and this person would be excluded from future phone based campaign lists.</p>
<br />
<p><span class="colorme">Display Name</span> is the name you would like to appear on email and mail lists.  Display name is also used to help find duplicates.  It is an important data point.</p>
<br />
<p>Just as important as display name is <span class="colorme">email</span>.  Email is used for finding duplicates and is one of our more important communication methods.</p>
<br />
<p>Make sure to select the correct <span class="colorme">preferred contact information.</span></p>
<br />
<p>Select the custom member fields, lists, and groups that apply to your organization''s business logic.</p>'
),

('groups',
'<p><span class="colorme">Groups are a way to describe a constituent</span>.  Group management provides a powerful tool for segmenting constituents for sophisticated reporting and campaigns.  They have a similar functionality to sets, labels or tags that you may have seen in other programs.</p><br />

<p>A constituent can be a  member of zero, one or many groups.</p><br />

<p><span class="colorme">There are two general types of groups</span>.  “Groups” and “group lists”.  Group lists are used for communication.  The naming convention is to not use white space and all lowercase letters.  Groups start with “group_” and lists start with “group_list_”.  Here are some examples.</p><br />

Groups:
<ul>
<li>group_donor</li>
<li>group_staff</li>
<li>group_volunteer</li>
</ul>

Group Lists:
<ul>
<li>group_list_newsletter</li>
<li>group_list_annual_appeal_email</li>
<li>group_list_annual_appeal_mail</li>
</ul>

<p>Groups can be modified in the settings section of the website.</p><br />

<p>Constituents can be assigned groups on the create and modify pages.</p><br />'
),

('campaigns',
'<span class="colorme">A campaign is when you desire to interact with multiple constituents</span>.  Usually this would be a communication (email, mail, telephone) or maybe an event.  Reports can be generated using campaign ID''s.<br /><br />
<p>
<b>Overview of a campaign.</b>
</p>
1.  <span class="colorme">Create a campaign name</span>.  It the settings menu you can create your campaign name.  An example name could be “2013 Annual Appeal”.  2013 Annual Appeal is now available in the drop down menu on the interactions page and the manage a campaign page.
<br /><br />
2.  <span class="colorme">Add people to your campaign and clean data</span>.  This can be done on the manage a campaign page.  You can add and subtract groups and individuals.  Prior to launching your campaign you can export it to CSV and verify constituent contact information is correct.
<br /><br />
3.  <span class="colorme">Launch your campaign</span>.  In settings on the manage a campaign page, you can export for labels or email.  This will create interactions for you.
<br /><br />
4.  <span class="colorme">Add information via interactions to your campaign</span>.  While constituents react to your campaign (RSVP, donate, volunteer, request to be taken off list, etc.) you can enter in information on the interactions page and associate the interactions with this campaign by selecting the appropriate name via the drop down menu.
<br /><br />
5.  <span class="colorme">Report on the campaign and analyze results</span>.  Using the campaign name we can export all interactions in to a spreadsheet giving staff the ability to measure performance and refine future campaigns.'
);

/*
here is where i'm putting all of my stored procedures.  the plan is
that eventually all queries will be stored procedures.  it will make life
much easier for front end designers and the api
*/

-- this is needed for storing passwords.  makes a bunch of functions.
-- note that this cannot be installed here.  user must do it manually.
--CREATE EXTENSION pgcrypto;

-- --------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION mneme_last_10_case_notes(userid int) RETURNS SETOF constituents_interactions AS
$BODY$
DECLARE
    r case_notes%rowtype;
BEGIN
    FOR r IN select * from case_notes where case_notes.user_id = $1 order by int_date desc limit 10
    LOOP
        RETURN NEXT r;
    END LOOP;
    RETURN;
END
$BODY$
LANGUAGE 'plpgsql' ;

-- --------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION mneme_case_notes_users ()
RETURNS SETOF case_notes_users AS
$BODY$

DECLARE
    r case_notes_users%rowtype;
BEGIN
    FOR r IN select * from case_notes_users
    LOOP
        RETURN NEXT r;
    END LOOP;
    RETURN;
END
$BODY$
LANGUAGE 'plpgsql' ;

-- --------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION mneme_last_10_interactions(userid int) RETURNS SETOF constituents_interactions AS
$BODY$
DECLARE
    r constituents_interactions%rowtype;
BEGIN
    FOR r IN select * from constituents_interactions where user_id = $1 order by int_date desc limit 10
    LOOP
        RETURN NEXT r;
    END LOOP;
    RETURN;
END
$BODY$
LANGUAGE 'plpgsql' ;

-- --------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION mneme_case_notes_process(
/*
the goal of this function is to process requests for creating, updating, and 
deleting case notes.  it will return true (1) on success and false (0) on failure.  usage:

select * from mneme_case_notes_process('process_action', 'v_note_id', 'v_user_id', 'v_int_date', 'v_int_notes', 'v_campaign_id');

select * from mneme_case_notes_process('action_create', '1', '2', '2013-09-16', 'First create test.', '2013_test');

process_action  -- function specific.  values:  action_create, action_delete, action_modify. mandatory
v_note_id  -- table specific. value ignored for action_create.
v_user_id  -- table specific. value ignored for action_delete and action_modify.
v_int_date  -- table specific.  value ignored for action_delete.
v_int_notes  -- table specific.  value ignored for action_delete.
v_campaign_id  -- table specific.  value ignored for action_delete.
*/

        process_action varchar,
        v_note_id integer,
        v_user_id integer,
        v_int_date date,
        v_int_notes varchar,
        v_campaign_id varchar

	) RETURNS boolean AS

$BODY$

DECLARE
mneme_case_notes_result boolean;  -- 1 = success, 0 = fail
v_user_id_check constituents_members.user_id%TYPE;

BEGIN

    CASE process_action

    WHEN 'action_create' THEN
        -- TODO:  check if user_id exists prior to update in a large multi-user 
        -- environment this could be a problem

        -- check if vars are supplied correctly
        IF v_int_notes IS NULL OR v_int_notes = '' OR v_int_notes = ' ' OR
        v_campaign_id IS NULL OR v_campaign_id = '' OR v_campaign_id = ' '
        THEN
        mneme_case_notes_result = 0; -- false
        ELSE  -- process request
        insert into case_notes (user_id, int_date, int_notes, campaign_id)
        values (v_user_id, v_int_date, v_int_notes, v_campaign_id);
        mneme_case_notes_result = 1; -- true
        END IF;

    WHEN 'action_delete' THEN
        -- TODO:  check if note_id exists prior to delete
        delete from case_notes where note_id = v_note_id;
        mneme_case_notes_result = 1; -- true

    WHEN 'action_modify' THEN

        -- TODO:  check if user_id exists prior to update in a large multi-user 
        -- environment this could be a problem

        -- check if vars are supplied correctly
        IF v_int_notes IS NULL OR v_int_notes = '' OR v_int_notes = ' ' OR
        v_campaign_id IS NULL OR v_campaign_id = '' OR v_campaign_id = ' '
        THEN
        mneme_case_notes_result = 0; -- false
        ELSE  -- process request
        update case_notes set (int_date, int_notes, campaign_id)
        = (v_int_date, v_int_notes, v_campaign_id)
        where v_note_id = note_id;
        mneme_case_notes_result = 1; -- true
        END IF;

    -- if we end up here they did not supply variable process_action
    ELSE  mneme_case_notes_result = 0;
    END CASE;

RETURN mneme_case_notes_result;

END
$BODY$
LANGUAGE 'plpgsql';


-- --------------------------------------------------------------------------

