/****************************************************************************
 ** Copyright (C) 2006  Xavier Cremaschi (omega.xavier@gmail.com)
 ** This file is part of teXswitcher.
 ** teXswitcher is a LaTeX document modifier.
 **
 ** teXswitcher is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License along
 ** with this program; if not, write to the Free Software Foundation, Inc.,
 ** 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 ****************************************************************************/


#include "Node.h"
#include "Keyword.h"
#include "Parser.h"

#include <iostream>
using namespace std;


Node::Node(string _body) : father(NULL), body(_body), footer("") {
}

Node::~Node() {
    vector<Node*>::iterator it;
    for (it = this->sons.begin() ; it != this->sons.end() ; it++) {
        delete *it;
    }
}

/**  add a son to this node */
void Node::add_son(Node* t) {
    t->father = this;
    this->sons.push_back(t);
}

bool Node::has_son(void) const {
    return (!this->sons.empty());
}


/** to remove a son we use remove_if */
// first a little class with the good operator()
// NB : could have been template...
class FindMatchingNode {
    private:
        const Node* to_find;
    public:
        FindMatchingNode(const Node* n) : to_find(n) { }
        bool operator() (const Node* n) const {
            return n == to_find;
        }
};

// then the remove_son method
void Node::remove_son(const Node* to_find) {

    // we search it in the vector
    vector<Node*>::iterator it = std::remove_if(this->sons.begin()
            , this->sons.end()
            , FindMatchingNode(to_find)); // call FindMatchingNode constructor with to_find


    // we remove it from the vector (object is not deleted)
    this->sons.erase(it, this->sons.end());



    // we shrink the vector with the "swap trick"
    // explanation: 
    // * the left of this expression (until the .) creates a copy of
    //   this->sons in a (unammed) temporary variable. It do not copy empty
    //   elements, so it is smaller than this->sons
    // * the right of this expression (after the .) calls swap on our temporary
    //   vector, so it takes the place of this->sons
    vector<Node*>(this->sons).swap(this->sons);
}





/**  return the ni son */
Node* Node::get_son(int i) const {
    if (0 <= i && i < static_cast<int>(this->sons.size())) {
        return this->sons[i];
    }
    else {
        return NULL;
    }
}

/**  search a son in this->sons and return its position */
int Node::get_index_of_son(const Node* son) const {
    for (int i = 0 ; i < static_cast<int>(this->sons.size()) ; i++) {
        if (this->sons[i] == son) {
            return i;
        }
    }

    return 0;
}


/**  return the node of the parent */
Node* Node::get_parent(void) const {
    return this->father;
}

/**  compute the number of node under this node (including this one) */
int Node::get_nb_of_nodes(void) const {
    int cpt = 1;
    for (unsigned int i = 0 ; i < this->sons.size() ; i++) {
        cpt += this->sons[i]->get_nb_of_nodes();
    }
    return cpt;
}

/**
 * search the lowest keyword under this node (including this one)
 */
int Node::get_sons_lowest_keyword_value(void) const {
    if(!this->has_son()) {
        return this->get_keyword_value();
    }
    else { // check all the subtree
        int min = this->sons[0]->get_sons_lowest_keyword_value();
        for (unsigned int i = 1 ; i < this->sons.size() ; i++) {
            int min_subtree = this->sons[i]->get_sons_lowest_keyword_value();
            min = (min_subtree < min) ? min_subtree : min;
        }
        return min;
    }
}

/**
 * search the biggest keyword under this node (excluding this one)
 * PRECONDITION : there is at least one son
 */
int Node::get_sons_biggest_keyword_value(void) const {
    int max = this->sons[0]->get_keyword_value();
    for (unsigned int i = 1 ; i < this->sons.size() ; i++) {
        int max_son = this->sons[i]->get_keyword_value();
        max = (max_son > max) ? max_son : max;
    }
    return max;
}

/**  the place of this node in his father's list */
int Node::get_number(void) const {
    int position = 0;
    if (this->father != NULL) {
        position = this->father->get_index_of_son(this);
    }
    return position;
}



/**  
 * return the keyword associated with a block, ie the substring between 
 * the begining of the string and the first { 
 * example :  
 * input    = \section{foobar}foofoofoofoo... 
 * output   = \section{ 
 */
string Node::get_keyword(void) const {
    string result = "";
    string::size_type begin = this->body.find("\\");
    if (begin != string::npos) {
        string::size_type end = this->body.find("{", begin) + 1;
        if (end != string::npos) {
            result = this->body.substr(begin, end-begin);
        }
    }
    return result;
}

/** return the value of the keyword (used to hierarchize keywords) */
int Node::get_keyword_value(void) const {
    return Keyword::value(this->get_keyword());
}


/** replace the string of the old keyword with new one */
void Node::change_keyword(int count, bool recursive) {
    int current_value = this->get_keyword_value();
    int new_value = current_value + count;
    if (Keyword::MIN_VALUE() <= new_value 
            && new_value < Keyword::MAX_VALUE()) {

        string new_keyword = Keyword::KEYWORD(new_value);

        // string substitution
        string::size_type begin = this->body.find("\\");
        if (begin != string::npos) {
            string::size_type end = this->body.find("{", begin) + 1;
            if (end != string::npos) {
                this->body.replace(begin, end, new_keyword);
            }
        }

        // repercussion to the sons
        if (recursive) {
            for (unsigned int i = 0 ; i < this->sons.size() ; i++) {
                this->sons[i]->change_keyword(count, recursive);
            }
        }
    }
}


/**  
 * return the title of the block 
 *  example :  
 *  input    = \section{foobar}foofoofoofoo... 
 *  output   = foobar 
 */
string Node::get_name(void) const {
    string result = "";
    string::size_type begin = this->body.find("{") + 1;
    if (begin != string::npos) {
        string::size_type end = this->body.find("}", begin);
        if (end != string::npos) {
            result = this->body.substr(begin, end-begin);
        }
    }
    return result;
}



/** footer will be added at the end of the string 
 *  returned by get_latex_document
 */
void Node::set_footer(string s) {
    this->footer = s;
}




/** 
 * serialization : build a string from a tree
 * NB : return the latex document if call with the root node 
 */
string Node::to_string(void) const {
    string result = this->body;

    for (unsigned int i = 0 ; i < sons.size() ; i++) {
        result += sons[i]->to_string();
    }

    result += this->footer;
    return result;
}

/** build a tree from a string */
Node* Node::from_string(string s) {
    return Parser::parse(s);
}



