#!/usr/bin/env perl
# Terminocheck, A Perl script which lets you check and rectify
# terminological mistakes such as using Linux instead of GNU/Linux
#
# Copyright 2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
# 
# This file is part of Terminocheck.
# 
# Terminocheck is free software: you can redistribute it and/or modify
# under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# Terminocheck is distributed in the hope that it will be useful,
# WITHOUT ANY WARRANTY; without even the implied warranty of
# or FITNESS FOR A PARTICULAR PURPOSE.  See the
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# with Terminocheck.  If not, see <http://www.gnu.org/licenses/>.
#

# Modules
use strict;
use warnings;
use Getopt::Long;

# Global variable (is it the "clean" way ?)
our $verbose = 0;

################
## SUBROUTINS ##
################

# sub askUser ($actions, $details)
# Print a pretty [detailed] prompt asking to choose an action to do
# Arguments :
#    $actions,
#    $goodTerm,
#    $details
# Return value (=answer) :
#    <STDIN> user input
sub askUser {
    my ($defaultAction, $goodTerm, $details) = @_;
    my $key;
    my $actions = "R/K/D/E/?/Q";
    $actions =~ s/$defaultAction/\[$defaultAction\]/;
    my %option = ("R" => "Replace this term by \"$goodTerm\"",
		  "K" => "Keep this occurence",
		  "D" => "Display more context (using grep -C)",
		  "E" => "Edit this file in your preferred editor (set in \$EDITOR)",
		  "?" => "Print this message again",
		  "Q" => "Quit the program here");
    # Detailed options information
    if ($details) {
	$actions =~ s#/?/#/#;
	print STDOUT ("These are the accepted options:\n");
	foreach $key (keys (%option)) {
	    print STDOUT ($key." :\t".$option{$key}.(($key eq $defaultAction)?"   (default)\n":"\n"));
	}
    }
    print STDOUT ("\e[38;5;46m==>\e[0m What is your choice ? ".$actions.": ");
    return <STDIN>;
}

# sub fancyDisplay ($beg, $mid, $end)
# Display a [colorfull] line of text
# Arguments :
#    $realMotif,
#    $motifToDisplay,
#    $message,
#    $code,
#    $file,
#    $line,
#    $color
sub fancyDisplay {
    my ($realMotif, $motifToDisplay, $message, $code,
	$file, $line, $color)
	= @_;
    my $prefix = ($realMotif ne $motifToDisplay) ? "New line:\t " : "";
    $message =~ /^.*?(.{0,30}?)((?:GNU\/)?$realMotif)(.{0,20}).*?$/i;
    # Colorized mode
    if (-t STDOUT && $color != 0) {
	print STDOUT ("\n$code==>\e[0m In file ".$file." line ".$line.
		      " :\t\n".$prefix."...".$1."\e[1m".$motifToDisplay.
		      "\e[0m".$3."...\n\n");
    }
    # Sad mode
    else {
	print STDOUT ("\n==> In file ".$file." line ".$line.
		      " :\t".$1.$2.$3."\n\n");
    }
}

# sub getFilesList ($path)
# Scan recursively a directory to get all its files
# TODO: check if files are readable
# Arguments :
#    $path of the directory
# Return value :
#    @filesList containing all the files
sub getFilesList {
    my ($fhRep,
	$fileFound,
	$path,
	@filesList,
	@content);
    
    $path = shift;
    opendir ($fhRep, $path)
	or die "Cannot open $path\n";
    @content = grep { !/^\.\.?$/ } readdir($fhRep);
    closedir ($fhRep);
    
    foreach $fileFound (@content) {
	# is a file
	if ( -f "$path/$fileFound") {
	    print ($fileFound." is a file\n") if ($verbose > 0);
	    push ( @filesList, "$path/$fileFound" );
	}
	# is a directory
	elsif ( -d "$path/$fileFound") {
	    print ($fileFound." is a directory\n") if ($verbose > 0);
	    # recall myself
	    push ( @filesList, getFilesList("$path/$fileFound"));
	}
    }
    return @filesList;
}

# sub prompt ($defaultAction, $details, $file, $line)
# Display the prompt, parse the user's answer
# Arguments :
#    $defaultAction,
#    $goodTerm,
#    $file,
#    $line
# Return values (=action to do) :
#   -1 for 'quit the program',
#    0 for 'skip line' (because don't want to replace),
#    1 for 'replace',
#    2 for 'skip file' (because has been edited)
sub prompt {
    my ($defaultAction, $goodTerm, $file, $line, $details) = @_;
    while ($_ = askUser ($defaultAction, $goodTerm, $details)) {
	$details = 0;
	$_ = $defaultAction."\n" if (/^\n$/);  # No answer: default action
	print STDERR ("you sayed $_") if ($verbose == 1);
	if (/^R\n$/i) {	                       # Replace: quit the sub with value 1
	    return 1;
	} elsif (/^K\n$/i) {	               # Keep: quit the sub with value 0
	    return 0;
	} elsif (/^D\n$/i) {	               # Display: invoke grep (and stay)
	    my $protectedline = protect ($line);
	    chomp ($protectedline);
	    my $command = "grep -C2 '".$protectedline."' $file";
	    print STDERR ("command: $command\n");
	    system ($command);
	} elsif (/^E\n$/i) {	               # Editor: invoke editor & quit with value 2
	    my $command = $ENV{'EDITOR'} ? $ENV{'EDITOR'}." ".$file : "nano ".$file;
	    print STDOUT ("Executing $command. Be careful, the file will be skipped,\n" .
			  "so edit every occurence yourself\n") if ($verbose > 0);
	    system ("sleep 1");
	    system ($command);
	    return 2;
	} elsif (/^\?\n$/i) {	               # ?: active details (and stay)
	    $details = 1;
	} elsif (/^Q\n$/i) {	               # Quit: quit the sub with value -1
	    return -1;
	} else {	                       # Answer not authorized: stay and active details
	    chomp ();
	    print STDERR ("Error, \"$_\" is not a good choice.\n" .
		   "You have to answer 'R', 'K', 'D', 'E' or 'Q'.\n");
	    $details = 1;
	}
    }
}

# sub protect ($string)
# Replace every double/single quotes in a string by a point '.'
# Arguments :
#    $string to process
# Return value :
#    $string processed
sub protect {
    my $string = shift;
    $string =~ s/'/\./g;
    $string =~ s/"/\./g;
    return $string;
}

# sub version ($number)
# Display the name of the script, it's version, a short description
# and copyright/license information, then quit the program.
sub version {
    my $number = 0.4.1; # Static (until better)
    print STDOUT ("Terminocheck version $number\n" .
		  "A Perl script which lets you check and rectify terminolo" .
		  "gical mistakes\nsuch as using Linux instead of GNU/Linux" .
		  ".\nCopyright 2016 (C) Felicien PILLOT <felicien.pillot\@" .
		  "member.fsf.org>\nTerminocheck is free software: you can " .
		  "redistribute it and/or modify\nunder the terms of the GN" .
		  "U General Public License as published by\nthe Free Softw" .
		  "are Foundation, either version 3 of the License, or\n(at" .
		  " your option) any later version.\n");
    exit (0);
}

# sub usage ()
# Display a text explaining the usage of the script, its options, etc.
# then quit the program.
sub usage {
    my $command = $0;
    $command =~ s#^.*/##;
    print STDERR ("usage: $command [OPTIONS] [FILES]\n" .
		  "Mandatory arguments to long options are mandatory for sh" .
		  "ort options too.\n  -b, --bad-term=STRING\t\tSet the wro" .
		  "ng/inappropriated term to STRING, default value is 'linu" .
		  "x'\n      --no-color\t\tDisable the colorized output\n  " .
		  "-g, --good-term=STRING\tSet the right/appropriated" .
		  " term to STRING, default value is 'GNU/Linux'\n  -h, --h" .
		  "elp\t\t\tDisplay this help and exit\n      --no-confirm " .
		  "\t\tDo not ask confirmation (use it carefully)\n  -r, --" .
		  "recursive=DIR\t\tProcess all files recursively in DIR\n " .
		  " -v, --verbose\t\t\tIncrease verbosity (useful informati" .
		  "on about process\t\t\n      --version\t\t\tDisplay versi" .
		  "on and copyright information and exit\n");
    exit (0);
}

################ 
##    MAIN    ##
################

# Declare a lot of variables
#### TODO: check if all of this is necessary
my ($answer,         # User's input
    $file,           # File given as argument
    $line,           # Current line examinated
    $next,           # Next lines (to display context)
    $nocolor,        # With this options, no color will be output
    $occurence,      # Badterm found and to replace
    $oldline,        # All is in the name
    $shrtbegin,      # Short display for $beginning
    $shrtend,        # Short display for $ending
    $total,          # File contents (modified or not) to write
    @filesList);     # List of files to parse

# Initialize some variables to their default values
my $badTerm = "linux";
my $color = 1;
my $confirm = 1;
my $defaultAction = "R";
my $details = 1;
my $goodTerm = "GNU/Linux";
my $recursive = "";

# Parse any command line arguments
Getopt::Long::Configure(qw{no_auto_abbrev bundling no_ignore_case_always});
Getopt::Long::GetOptions('b|bad-term=s'    => \$badTerm,
			 'color!'          => \$color,
			 'confirm!'        => \$confirm,
			 'g|good-term=s'   => \$goodTerm,
			 'h|help|u|usage'  => \&usage,
			 'r|recursive=s'   => \$recursive,
			 'v|verbose+'      => \$verbose,
			 'version'         => \&version )
    or exit (0);

# Get the list of files from the command line arguments
if ($recursive) {
    if ( -d $recursive) {
	@filesList = getFilesList ($recursive);
    } else {
	print STDERR ("Error: $recursive is not a directory\n");
    }
} else {
    usage () if (!@ARGV);
    foreach $file (@ARGV) {
	if (! -f $file) {
	    print STDERR ("Error: $file is not a file.\n");
	} else {
	    push (@filesList, $file);
	}
    }
}

## Look for occurence and ask the user for each of these files
foreach $file (@filesList) {
    $total = "";
    open (FILE, "<", $file) or die "File $file unreadable (corrupted/bad " .
	"permissions)\n";
    while (<FILE>) {
	print ("Processing file ".$file."...\n") if ($verbose > 0);
	$line = $_;
	if (/((?<!GNU\/)$badTerm)/i) {
	    fancyDisplay ($badTerm, $badTerm, $line, "\e[38;5;1m",
			  $file, $., $color);
	    $answer = (($confirm == 0) ? 1 :
		       prompt ($defaultAction, $goodTerm, $file,
			       $line, $details));
	    # From now, hide the detailed informations
	    $details = 0;
	    # Quit if user tells us to quit
	    if ($answer == -1) {
		# Add the rest of the file (unmodified)
		$total .= $line;
		$total .= $_ while (<FILE>);
		close (FILE);
		# Write it, so the modifications in the beginning are saved
		open (FILE, ">", $file) or
		    die "File $file unwritable (corrupted/bad permissions)\n";
		print FILE ($total);
		close (FILE);
		# Quit the program
		print STDOUT ("Program ended, and your changes" .
			      " have been saved\n");
		exit (0);
	    }
	    if ($answer == 0) {
		$defaultAction = "K";
	    } elsif ($answer == 1) {
		$oldline = $line;
		### FINALLY REPLACE THIS @!&#*$ OCCURENCE !!! :D ###
		$line = $`.$goodTerm.$';
		$defaultAction = "R";
		fancyDisplay ($badTerm, $goodTerm, $oldline, "\e[38;5;21m",
			      $file, $., $color);
	    } elsif ($answer == 2) {
		$defaultAction = "E";
		last;
	    } else { print "Error !!"; }
	}
	# Add this line to the file
	$total .= $line;
    }
    close (FILE);
    # Write modifications unless the file has already been edited
    unless ($defaultAction eq "E") {
	open (FILE, ">", $file) or die "File $file unwritable (corrupted/bad permissions)\n";
	print FILE ($total);
	close (FILE);
    }
}

# End of the program
print STDOUT ("Program ended succesfully. You may want to re-run it to\n" .
	      "check if there isn't double occurences in same lines, in any file\n");
