/* $Id: check_fixedlenio.c 659 2006-05-13 14:51:08Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "baseio.h"
#include "fixedlenio.h"
#include "check_io.h"
#include "check_fixedlenio.h"

#define TEST_FILE "/tmp/bumtest.fixedlenio.tmp"
#define TEST_FIXED_SIZE (1024 * 1024)

static int
read_write_test_size ()
{
  switch (get_test_size ())
    {
    case QUICK_TEST:
      return 256;
    case NORMAL_TEST:
      return 1024;
    case LONG_TEST:
      return 10240;
    default:
      fail ("Invalid test size");
      return -1;                // Not reached, prevent compiler warning
    }
}

START_TEST (check_out_basic)
{
  out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
  fail_if (NULL == base_outs);

  out_stream_t outs = fixedlenio_open_out (base_outs, TEST_FIXED_SIZE);
  fail_if (NULL == outs);
  basic_out_test (outs);

  release_out (outs);
  fail_unless (OUTPUT_OK == close_out (base_outs));
  release_out (base_outs);
}
END_TEST static void
check_read_write (int block_size)
{
  const int
    test_size = read_write_test_size (), count = test_size / block_size;

  assert (count > 0);
  assert (test_size > 0);

  out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
  fail_if (NULL == base_outs);

  out_stream_t outs = fixedlenio_open_out (base_outs, TEST_FIXED_SIZE);
  fail_if (NULL == outs);

  ascending_output_half_test (false, outs, block_size, count);
  release_out (outs);
  fail_unless (OUTPUT_OK == close_out (base_outs));
  release_out (base_outs);

  mark_point ();

  in_stream_t base_ins = baseio_open_in (TEST_FILE, "r");
  fail_if (NULL == base_ins);

  in_stream_t ins = fixedlenio_open_in (base_ins, TEST_FIXED_SIZE);
  fail_if (NULL == ins);

  ascending_input_half_test (false, false, ins, block_size, count);
  release_in (ins);

  fail_unless (INPUT_OK == close_in (base_ins));
  release_in (base_ins);
}

START_TEST (check_read_write_small_blocks)
{
  check_read_write (1);
}

END_TEST
START_TEST (check_read_write_medium_blocks)
{
  check_read_write (32);
}

END_TEST
START_TEST (check_read_write_large_blocks)
{
  check_read_write (256);
}

END_TEST
START_TEST (check_limits)
{
  {
    out_stream_t base_outs = baseio_open_out (TEST_FILE, "w");
    fail_if (NULL == base_outs);

    out_stream_t outs = fixedlenio_open_out (base_outs, TEST_FIXED_SIZE);
    fail_if (NULL == outs);

    char buf = 'J';
    iobuffer_t iobuf;
    for (int i = 0; i < TEST_FIXED_SIZE; i++)
      {
        init_iobuffer_with (&iobuf, 1, 1, &buf);
        fail_unless_out_status (OUTPUT_OK, output_all (outs, &iobuf));
      }

    init_iobuffer_with (&iobuf, 1, 1, &buf);
    fail_unless (OUTPUT_ERR_REFUSED == output_all (outs, &iobuf));

    fail_unless (OUTPUT_OK == close_out (outs));
    release_out (outs);

    fail_unless (OUTPUT_OK == close_out (base_outs));
    release_out (base_outs);
  }

  mark_point ();

  {
    in_stream_t base_ins = baseio_open_in (TEST_FILE, "r");
    fail_if (NULL == base_ins);

    in_stream_t ins = fixedlenio_open_in (base_ins, (TEST_FIXED_SIZE / 2));
    fail_if (NULL == ins);

    char buf;
    iobuffer_t iobuf;

    for (int i = 0; i < (TEST_FIXED_SIZE / 2); i++)
      {
        init_iobuffer_with (&iobuf, 1, 0, &buf);
        fail_unless_in_status (INPUT_OK, input_all (ins, &iobuf));
      }

    init_iobuffer_with (&iobuf, 1, 0, &buf);
    fail_unless (INPUT_EOF == input_all (ins, &iobuf));

    fail_unless (INPUT_OK == close_in (ins));
    release_in (ins);

    fail_unless (INPUT_OK == close_in (base_ins));
    release_in (base_ins);
  }
}
END_TEST void
add_fixedlenio_tests (Suite * s)
{
  TCase *tc_core = tcase_create ("FixedLenIO");
  tcase_set_timeout (tc_core, 60);
  tcase_add_test (tc_core, check_out_basic);
  tcase_add_test (tc_core, check_read_write_small_blocks);
  tcase_add_test (tc_core, check_read_write_medium_blocks);
  tcase_add_test (tc_core, check_read_write_large_blocks);
  tcase_add_test (tc_core, check_limits);

  suite_add_tcase (s, tc_core);
}
