/* $Id: check_deque.c 659 2006-05-13 14:51:08Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <assert.h>

#include "check_io.h"
#include "check_deque.h"

#include "deque.h"

// TODO: Seperate tests so they test specific functions

static int
add_remove_test_size ()
{
  switch (get_test_size ())
    {
    case QUICK_TEST:
      return 256;
    case NORMAL_TEST:
      return 1024;
    case LONG_TEST:
      return 1024 * 1024;
    default:
      fail ("Invalid test size");
      return -1;                // Not reached, prevent compiler warning
    }
}

static void
check_deque_add_remove (const int test_size,
                        const int block_size,
                        const int add_size, const int remove_size)
{
  assert (remove_size <= add_size);

  int added = 0, removed = 0;

  deque_t deque = create_deque (sizeof (int), block_size);
  fail_if (!deque);

  while (removed < test_size)
    {

      // Do a sequence of adds
      if (added < test_size)
        {
          for (int i = 0; i < add_size; i++)
            {
              if (added < test_size)
                {
                  fail_unless (single_add_to_deque_end (deque, &added));
                  added++;

                  if (added == removed)
                    fail_unless (is_deque_empty (deque));
                  else
                    fail_if (is_deque_empty (deque));
                }
            }
        }

      // Do a sequence of removes
      for (int i = 0; i < remove_size; i++)
        {
          if (removed < test_size)
            {
              int buf;
              fail_unless (single_remove_from_deque_start (deque, &buf));
              if (buf != removed)
                {
                  char msg[256];
                  sprintf (msg, "Expected %d but got %d", removed, buf);
                  fail (msg);
                }
              removed++;

              if (added == removed)
                fail_unless (is_deque_empty (deque));
              else
                fail_if (is_deque_empty (deque));
            }
        }

    }

  release_deque (deque);
}

static void
check_deque_add_remove_reverse (const int test_size,
                                const int block_size,
                                const int add_size, const int remove_size)
{
  assert (remove_size <= add_size);

  int added = 0, removed = 0;

  deque_t deque = create_deque (sizeof (int), block_size);
  fail_if (!deque);

  while (removed < test_size)
    {
      // Do a sequence of adds
      if (added < test_size)
        {
          for (int i = 0; i < add_size; i++)
            {
              if (added < test_size)
                {
                  fail_unless (single_add_to_deque_start (deque, &added));
                  added++;

                  if (added == removed)
                    fail_unless (is_deque_empty (deque));
                  else
                    fail_if (is_deque_empty (deque));
                }
            }
        }

      // Do a sequence of removes
      for (int i = 0; i < remove_size; i++)
        {
          if (removed < test_size)
            {
              int buf;
              fail_unless (single_remove_from_deque_end (deque, &buf));
              if (buf != removed)
                {
                  char msg[256];
                  sprintf (msg, "Expected %d but got %d", removed, buf);
                  fail (msg);
                }
              removed++;
              if (added == removed)
                fail_unless (is_deque_empty (deque));
              else
                fail_if (is_deque_empty (deque));
            }
        }
    }

  release_deque (deque);
}

static void
check_deque_add_remove_chunk (const int test_length,
                              const int chunk_size,
                              const int block_size,
                              const int add_size, const int remove_size)
{
  assert (remove_size <= add_size);

  int added = 0, removed = 0;

  deque_t deque = create_deque (sizeof (int), block_size);
  fail_if (!deque);

  const int test_size = test_length / chunk_size;
  assert (test_size > 0);

  int data[chunk_size];
  while (removed < test_size)
    {
      // Do a sequence of adds
      if (added < test_size)
        {
          for (int i = 0; i < add_size; i++)
            {
              if (added < test_size)
                {
                  for (int j = 0; j < chunk_size; j++)
                    data[j] = added + j;
                  fail_unless (chunk_size ==
                               add_to_deque_end (deque, data, chunk_size));
                  added++;

                  if (added == removed)
                    fail_unless (is_deque_empty (deque));
                  else
                    fail_if (is_deque_empty (deque));
                }
            }
        }

      // Do a sequence of removes
      for (int i = 0; i < remove_size; i++)
        {
          if (removed < test_size)
            {
              // Blank the data so we don't think we have read it when
              // actually nothing has happened
              for (int j = 0; j < chunk_size; j++)
                data[j] = -1;

              fail_unless (remove_from_deque_start (deque, data, chunk_size));
              for (int j = 0; j < chunk_size; j++)
                {
                  if (data[j] != removed + j)
                    {
                      char errmsg[256];
                      sprintf (errmsg, "Expected %d but got %d (j=%d)",
                               removed + j, data[j], j);
                      fail (errmsg);
                    }
                }
              removed++;

              if (added == removed)
                fail_unless (is_deque_empty (deque));
              else
                fail_if (is_deque_empty (deque));
            }
        }
    }

  release_deque (deque);
}

static void
check_deque_add_remove_chunk_reverse (const int test_length,
                                      const int chunk_size,
                                      const int block_size,
                                      const int add_size,
                                      const int remove_size)
{
  assert (remove_size <= add_size);

  int added = 0, removed = 0;

  deque_t deque = create_deque (sizeof (int), block_size);
  fail_if (!deque);

  const int test_size = test_length / chunk_size;
  assert (test_size > 0);

  int data[chunk_size];
  while (removed < test_size)
    {
      // Do a sequence of adds
      if (added < test_size)
        {
          for (int i = 0; i < add_size; i++)
            {
              if (added < test_size)
                {
                  for (int j = 0; j < chunk_size; j++)
                    data[j] = added + j;
                  fail_unless (chunk_size ==
                               add_to_deque_start (deque, data, chunk_size));
                  added++;

                  if (added == removed)
                    fail_unless (is_deque_empty (deque));
                  else
                    fail_if (is_deque_empty (deque));
                }
            }
        }

      // Do a sequence of removes
      for (int i = 0; i < remove_size; i++)
        {
          if (removed < test_size)
            {
              // Blank the data so we don't think we have read it when
              // actually nothing has happened
              for (int j = 0; j < chunk_size; j++)
                data[j] = -1;

              fail_unless (chunk_size ==
                           remove_from_deque_end (deque, data, chunk_size));
              for (int j = 0; j < chunk_size; j++)
                {
                  if (data[j] != removed + j)
                    {
                      char errmsg[256];
                      snprintf (errmsg, 256,
                                "Expected %d but got %d (added=%d, removed=%d, j=%d)",
                                removed + j, data[j], added, removed, j);
                      fail (errmsg);
                    }
                }
              removed++;
              if (added == removed)
                fail_unless (is_deque_empty (deque));
              else
                fail_if (is_deque_empty (deque));
            }
        }
    }

  release_deque (deque);
}

START_TEST (check_deque_add_remove_1)
{
  check_deque_add_remove (add_remove_test_size (), 256, 1, 1);
}

END_TEST
START_TEST (check_deque_add_remove_2)
{
  check_deque_add_remove (add_remove_test_size (), 256, 2, 1);
}

END_TEST
START_TEST (check_deque_add_remove_3)
{
  check_deque_add_remove (add_remove_test_size (), 256, 128, 1);
}

END_TEST
START_TEST (check_deque_add_remove_4)
{
  check_deque_add_remove (add_remove_test_size (), 256, 128, 2);
}

END_TEST
START_TEST (check_deque_add_remove_5)
{
  check_deque_add_remove (add_remove_test_size (), 256, 128, 128);
}

END_TEST
START_TEST (check_deque_add_remove_reverse_1)
{
  check_deque_add_remove_reverse (add_remove_test_size (), 256, 1, 1);
}

END_TEST
START_TEST (check_deque_add_remove_reverse_2)
{
  check_deque_add_remove_reverse (add_remove_test_size (), 256, 2, 1);
}

END_TEST
START_TEST (check_deque_add_remove_reverse_3)
{
  check_deque_add_remove_reverse (add_remove_test_size (), 256, 128, 1);
}

END_TEST
START_TEST (check_deque_add_remove_reverse_4)
{
  check_deque_add_remove_reverse (add_remove_test_size (), 256, 128, 2);
}

END_TEST
START_TEST (check_deque_add_remove_reverse_5)
{
  check_deque_add_remove_reverse (add_remove_test_size (), 256, 128, 128);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_1)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 2, 256, 1, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_2)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 2, 256, 2, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_3)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 7, 256, 1, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_4)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 7, 256, 2, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_5)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 256, 256, 1, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_6)
{
  check_deque_add_remove_chunk (add_remove_test_size (), 256, 256, 2, 1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_1)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 2, 256, 1,
                                        1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_2)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 2, 256, 2,
                                        1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_3)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 7, 256, 1,
                                        1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_4)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 7, 256, 2,
                                        1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_5)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 256, 256, 1,
                                        1);
}

END_TEST
START_TEST (check_deque_add_remove_chunk_reverse_6)
{
  check_deque_add_remove_chunk_reverse (add_remove_test_size (), 256, 256, 2,
                                        1);
}
END_TEST void
add_deque_tests (Suite * s)
{
  TCase *tc_core = tcase_create ("Deque");
  tcase_add_test (tc_core, check_deque_add_remove_1);
  tcase_add_test (tc_core, check_deque_add_remove_2);
  tcase_add_test (tc_core, check_deque_add_remove_3);
  tcase_add_test (tc_core, check_deque_add_remove_4);
  tcase_add_test (tc_core, check_deque_add_remove_5);
  tcase_add_test (tc_core, check_deque_add_remove_reverse_1);
  tcase_add_test (tc_core, check_deque_add_remove_reverse_2);
  tcase_add_test (tc_core, check_deque_add_remove_reverse_3);
  tcase_add_test (tc_core, check_deque_add_remove_reverse_4);
  tcase_add_test (tc_core, check_deque_add_remove_reverse_5);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_1);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_2);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_3);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_4);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_5);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_6);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_1);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_2);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_3);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_4);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_5);
  tcase_add_test (tc_core, check_deque_add_remove_chunk_reverse_6);
  suite_add_tcase (s, tc_core);
}
